/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.product;

import nextapp.echo2.app.event.WindowPaneEvent;
import org.apache.commons.collections.Predicate;
import org.openvpms.archetype.rules.product.ProductSupplier;
import org.openvpms.archetype.rules.supplier.SupplierRules;
import org.openvpms.component.business.domain.im.common.IMObjectReference;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.business.service.archetype.functor.NodeEquals;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.bean.Policies;
import org.openvpms.component.model.bean.Predicates;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.entity.EntityLink;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;
import org.openvpms.component.system.common.query.ArchetypeQueryException;
import org.openvpms.web.component.im.edit.AbstractIMObjectReferenceEditor;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.query.Browser;
import org.openvpms.web.component.im.query.BrowserDialog;
import org.openvpms.web.component.im.query.BrowserFactory;
import org.openvpms.web.component.im.query.DefaultIMObjectTableBrowser;
import org.openvpms.web.component.im.query.ListQuery;
import org.openvpms.web.component.im.query.Query;
import org.openvpms.web.component.im.util.IMObjectHelper;
import org.openvpms.web.component.im.view.TableComponentFactory;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.component.util.CollectionHelper;
import org.openvpms.web.echo.dialog.ConfirmationDialog;
import org.openvpms.web.echo.event.WindowPaneListener;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;

import java.util.List;
import java.util.Objects;


/**
 * Editor for product {@link IMObjectReference}s.
 *
 * @author Tim Anderson
 */
public class ProductReferenceEditor extends AbstractIMObjectReferenceEditor<Product> {

    /**
     * The parent editor.
     */
    private final ProductParticipationEditor editor;

    /**
     * The archetype service.
     */
    private final IArchetypeService service;

    /**
     * Constructs a {@link ProductReferenceEditor}.
     *
     * @param editor   the parent editor
     * @param property the product reference property
     * @param context  the layout context
     */
    public ProductReferenceEditor(ProductParticipationEditor editor, Property property, LayoutContext context) {
        super(property, editor.getParent(), new DefaultLayoutContext(context,
                                                                     context.getHelpContext().topic("product")));
        this.editor = editor;
        service = ServiceHelper.getArchetypeService();
    }

    /**
     * Invoked when an object is selected.
     *
     * @param product the selected object. May be {@code null}
     */
    @Override
    protected void onSelected(Product product) {
        if (product != null && editor.getSupplier() != null && hasSuppliers(product)) {
            checkSupplier(product);
        } else {
            setProduct(product, null);
        }
    }

    /**
     * Invoked when the underlying property updates.
     * This updates the product supplier relationship, unless there is already one present for the supplier.
     *
     * @param product the updated object. May be {@code null}
     */
    @Override
    protected void onUpdated(Product product) {
        if (product != null && hasSuppliers(product)) {
            ProductSupplier productSupplier = editor.getProductSupplier();
            Party supplier = editor.getSupplier();
            if (productSupplier == null || supplier == null
                || !Objects.equals(supplier.getObjectReference(), productSupplier.getSupplierRef())
                || !Objects.equals(product.getObjectReference(), productSupplier.getProductRef())) {
                List<EntityLink> relationships = getSupplierRelationships(product);
                if (relationships.isEmpty()) {
                    setProductSupplier(null);
                } else if (relationships.size() == 1) {
                    setProductSupplier(relationships.get(0));
                } else {
                    setProductSupplier(getPreferred(relationships));
                }
            }
        } else {
            setProductSupplier(null);
        }
    }

    /**
     * Creates a query to select objects.
     *
     * @param name the name to filter on. May be {@code null}
     * @return a new query
     * @throws ArchetypeQueryException if the short names don't match any
     *                                 archetypes
     */
    @Override
    protected Query<Product> createQuery(String name) {
        Query<Product> query = super.createQuery(name);
        return getQuery(query);
    }

    /**
     * Creates a new browser.
     *
     * @param query the query
     * @return a return a new browser
     */
    @Override
    protected Browser<Product> createBrowser(Query<Product> query) {
        ProductQuery q = (ProductQuery) query;
        LayoutContext context = getLayoutContext();
        ProductTableModel model = new ProductTableModel(q, editor.getLocation(), context);
        return new DefaultIMObjectTableBrowser<>(query, model, context);
    }

    /**
     * Constrains the query on species and stock location, if a patient and stock location is present.
     *
     * @param query the query
     * @return the query
     */
    protected Query<Product> getQuery(Query<Product> query) {
        if (query instanceof ProductQuery) {
            ProductQuery productQuery = ((ProductQuery) query);
            Party patient = editor.getPatient();
            if (patient != null) {
                String species = (String) IMObjectHelper.getValue(patient, "species");
                if (species != null) {
                    productQuery.setSpecies(species);
                }
            }
            Party location = editor.getStockLocation();
            productQuery.setUseLocationProducts(editor.useLocationProducts());
            productQuery.setLocation(editor.getLocation());
            productQuery.setStockLocation(location);
            productQuery.setExcludeTemplateOnlyProducts(editor.getExcludeTemplateOnlyProducts());
        }
        return query;
    }

    /**
     * Determines if the reference is valid.
     * <p/>
     * TODO - this is an expensive operation as products do filtering on species and stock location.
     * The check is disabled in 1.4 - needs to be enabled in 1.5 where there is better left join support
     *
     * @param validator the validator
     * @return {@code true} if the reference is valid, otherwise {@code false}
     */
    @Override
    protected boolean isValidReference(Validator validator) {
        return true;
    }

    /**
     * Updates the product details.
     *
     * @param product      the product. May be {@code null}
     * @param relationship the product supplier relationship. May be {@code null}
     */
    private void setProduct(Product product, EntityLink relationship) {
        setProductSupplier(relationship);
        setObject(product);
    }

    /**
     * Sets the product supplier relationship.
     *
     * @param relationship the relationship. May be {@code null}
     */
    private void setProductSupplier(EntityLink relationship) {
        if (relationship != null) {
            editor.setProductSupplier(new ProductSupplier(relationship, ServiceHelper.getArchetypeService()));
        } else {
            editor.setProductSupplier(null);
        }
    }

    /**
     * Checks if a product is supplied by a different supplier. If so,
     * pops up a dialog to cancel the selection. If there are no other
     * suppliers, or the selection isn't cancelled, invokes
     * {@link #checkProductSupplierRelationships}.
     *
     * @param product the product
     */
    private void checkSupplier(final Product product) {
        SupplierRules rules = ServiceHelper.getBean(SupplierRules.class);
        Entity otherSupplier;

        if (!rules.isSuppliedBy(editor.getSupplier(), product) && (otherSupplier = getSupplier(product)) != null) {
            String title = Messages.get("product.othersupplier.title");
            String message = Messages.format("product.othersupplier.message", product.getName(),
                                             otherSupplier.getName());
            ConfirmationDialog dialog = new ConfirmationDialog(title, message);
            dialog.addWindowPaneListener(new WindowPaneListener() {
                public void onClose(WindowPaneEvent event) {
                    if (ConfirmationDialog.OK_ID.equals(dialog.getAction())) {
                        checkProductSupplierRelationships(product);
                    } else {
                        // cancel the update
                        setProduct(null, null);
                    }
                }
            });
            dialog.show();
        } else {
            checkProductSupplierRelationships(product);
        }
    }

    /**
     * Determines if a product can have suppliers.
     *
     * @param product the product
     * @return {@code true} if the product can have suppliers, otherwise
     * {@code false}
     */
    private boolean hasSuppliers(Product product) {
        IMObjectBean bean = service.getBean(product);
        return bean.hasNode("suppliers");
    }

    /**
     * Returns the first active supplier for a product.
     *
     * @param product the product
     * @return the supplier, or {@code null} if none is found
     */
    private Entity getSupplier(Product product) {
        IMObjectBean bean = service.getBean(product);
        return bean.getTarget("suppliers", Entity.class, Policies.active());
    }

    /**
     * Determines if there is product supplier relationships for the current
     * supplier and specified product.
     * <p/>
     * If there is more than one, pops up a dialog prompting to select one of them.
     *
     * @param product the product
     */
    private void checkProductSupplierRelationships(final Product product) {
        // find all relationships for the product and supplier
        List<EntityLink> relationships = getSupplierRelationships(product);

        if (relationships.isEmpty()) {
            setProduct(product, null);
        } else if (relationships.size() == 1) {
            setProduct(product, relationships.get(0));
        } else {
            // pop up a browser displaying the relationships, with the preferred one selected
            EntityLink preferred = getPreferred(relationships);
            Query<EntityLink> query = new ListQuery<>(relationships, "entityLink.productSupplier", EntityLink.class);
            String title = Messages.get("product.supplier.type");
            LayoutContext context = new DefaultLayoutContext(getLayoutContext());
            context.setComponentFactory(new TableComponentFactory(context));
            Browser<EntityLink> browser = BrowserFactory.create(query, context);
            BrowserDialog<EntityLink> dialog = new BrowserDialog<>(title, browser, context.getHelpContext());

            dialog.addWindowPaneListener(new WindowPaneListener() {
                public void onClose(WindowPaneEvent event) {
                    EntityLink selected = browser.getSelected();
                    if (selected != null) {
                        setProduct(product, selected);
                    } else {
                        // cancel the update
                        setProduct(null, null);
                    }
                }
            });
            browser.query();
            browser.setSelected(preferred);
            dialog.show();
        }
    }

    /**
     * Returns the preferred supplier relationship.
     *
     * @param relationships the relationships
     * @return the preferred relationship, or the first if none is preferred,
     * or {@code null} if there are no relationships
     */
    private EntityLink getPreferred(List<EntityLink> relationships) {
        EntityLink result = null;
        if (!relationships.isEmpty()) {
            Predicate preferred = new NodeEquals("preferred", true, ServiceHelper.getArchetypeService());
            result = CollectionHelper.find(relationships, preferred);
            if (result == null) {
                result = relationships.get(0);
            }
        }
        return result;
    }

    /**
     * Returns all active supplier relationships for the current supplier
     * and specified product.
     *
     * @param product the product
     * @return the active relationships
     */
    private List<EntityLink> getSupplierRelationships(Product product) {
        IMObjectBean bean = service.getBean(product);
        Party supplier = editor.getSupplier();
        return bean.getValues("suppliers", EntityLink.class,
                              Predicates.<EntityLink>activeNow().and(Predicates.targetEquals(supplier)));
    }
}
