/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.edit.payment;

import org.openvpms.archetype.rules.math.Currency;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.web.component.app.ContextHelper;
import org.openvpms.web.component.im.edit.act.AbstractActEditor;
import org.openvpms.web.component.im.layout.AbstractLayoutStrategy;
import org.openvpms.web.component.im.layout.IMObjectLayoutStrategy;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.view.ComponentState;
import org.openvpms.web.component.property.MutablePropertySet;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.PropertySet;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.component.util.ErrorHelper;

import java.math.BigDecimal;


/**
 * An editor for {@link Act}s which have an archetype in
 * <em>act.customerAccountPayment*</em>, <em>act.customerAccountRefund*</em>
 * <em>act.supplierAccountPayment*</em> and <em>act.supplierAccountRefund*</em>
 *
 * @author Tim Anderson
 */
public abstract class PaymentItemEditor extends AbstractActEditor {

    /**
     * The amount node.
     */
    protected static final String AMOUNT = "amount";

    /**
     * Determines if the amount should be read-only.
     */
    private boolean readOnlyAmount;

    /**
     * The rounded amount node.
     */
    private static final String ROUNDED_AMOUNT = "roundedAmount";

    /**
     * The tendered node.
     */
    private static final String TENDERED = "tendered";

    /**
     * Constructs a {@link PaymentItemEditor}.
     *
     * @param act     the act to edit
     * @param parent  the parent act
     * @param context the layout context
     */
    public PaymentItemEditor(FinancialAct act, FinancialAct parent, LayoutContext context) {
        super(act, parent, context);
        if (act.isNew() && parent != null) {
            // default the act start time to that of the parent
            act.setActivityStartTime(parent.getActivityStartTime());
        }
        if (getProperty(ROUNDED_AMOUNT) != null) {
            // need to derive the rounded and tendered amounts from the amount
            Property amount = getProperty(AMOUNT);
            amount.addModifiableListener(modifiable -> onAmountChanged());
            if (act.isNew()) {
                onAmountChanged();
            }
        }
    }

    /**
     * Returns the payment amount.
     *
     * @return the payment amount
     */
    public BigDecimal getAmount() {
        return getProperty(AMOUNT).getBigDecimal(BigDecimal.ZERO);
    }

    /**
     * Sets the amount.
     *
     * @param amount the amount
     */
    public void setAmount(BigDecimal amount) {
        getProperty(AMOUNT).setValue(amount);
    }

    /**
     * Determines if the amount should be read-only.
     *
     * @param readOnly if {@code true} the amount is read-only to users
     */
    public void setReadOnlyAmount(boolean readOnly) {
        if (readOnlyAmount != readOnly) {
            readOnlyAmount = readOnly;
            if (getView().hasComponent()) {
                onLayout();
            }
        }
    }

    /**
     * Determines if this item can appear in a split payment or refund.
     *
     * @return {@code true} if the item can appear in a split transaction, {@code false} if it may be the only
     * item present in a transaction
     */
    public boolean supportsSplitTransactions() {
        return true;
    }

    /**
     * Determines if the editor can be deleted.
     *
     * @return {@code true} if the editor can be deleted
     */
    public boolean canDelete() {
        return true;
    }

    /**
     * Creates the layout strategy.
     *
     * @return a new layout strategy
     */
    @Override
    protected IMObjectLayoutStrategy createLayoutStrategy() {
        if (readOnlyAmount) {
            return new LayoutStrategy();
        }
        return super.createLayoutStrategy();
    }

    /**
     * Validates that the start and end times are valid.
     * <p/>
     * This implementation always returns {@code true}
     *
     * @param validator the validator
     * @return {@code true}
     */
    @Override
    protected boolean validateStartEndTimes(Validator validator) {
        return true;
    }

    /**
     * Invoked when the amount changes, to update the rounded and tendered
     * amount.
     * Only applies to cash payments.
     */
    private void onAmountChanged() {
        try {
            BigDecimal amount = (BigDecimal) getProperty(AMOUNT).getValue();
            BigDecimal rounded = amount;
            Property roundedAmount = getProperty(ROUNDED_AMOUNT);
            Currency currency = ContextHelper.getPracticeCurrency(
                    getLayoutContext().getContext());
            if (currency != null) {
                rounded = currency.roundCash(amount);
            }
            roundedAmount.setValue(rounded);
            Property tenderedAmount = getProperty(TENDERED);
            if (tenderedAmount != null) {
                tenderedAmount.setValue(rounded);
            }
        } catch (OpenVPMSException exception) {
            ErrorHelper.show(exception);
        }
    }

    private class LayoutStrategy extends AbstractLayoutStrategy {

        /**
         * Apply the layout strategy.
         * <p>
         * This renders an object in a {@code Component}, using a factory to create the child components.
         *
         * @param object     the object to apply
         * @param properties the object's properties
         * @param parent     the parent object. May be {@code null}
         * @param context    the layout context
         * @return the component containing the rendered {@code object}
         */
        @Override
        public ComponentState apply(IMObject object, PropertySet properties, IMObject parent, LayoutContext context) {
            if (readOnlyAmount) {
                MutablePropertySet set = new MutablePropertySet(properties);
                set.setReadOnly(AMOUNT);
                properties = set;
            }
            return super.apply(object, properties, parent, context);
        }
    }

}
