/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.alert;

import org.openvpms.archetype.rules.act.ActStatus;
import org.openvpms.archetype.rules.patient.PatientArchetypes;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.IMObjectQueryIterator;
import org.openvpms.component.system.common.query.JoinConstraint;
import org.openvpms.component.system.common.query.ParticipationConstraint;
import org.openvpms.web.component.im.query.QueryHelper;
import org.openvpms.web.component.prefs.UserPreferences;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import static org.openvpms.component.system.common.query.Constraints.eq;
import static org.openvpms.component.system.common.query.Constraints.join;
import static org.openvpms.component.system.common.query.ParticipationConstraint.Field.ActShortName;

/**
 * Patient alerts.
 *
 * @author Tim Anderson
 */
class PatientAlerts extends Alerts {

    /**
     * Constructs a {@link PatientAlerts}.
     *
     * @param preferences the preferences
     * @param service     the archetype service
     */
    public PatientAlerts(UserPreferences preferences, IArchetypeService service) {
        super(preferences, "patientAlerts", service);
    }

    /**
     * Returns alerts for a party.
     *
     * @param patient the party
     * @return the alerts for a party
     */
    @Override
    public List<Alert> getAlerts(Party patient) {
        List<Alert> result = new ArrayList<>();
        ArchetypeQuery query = new ArchetypeQuery(PatientArchetypes.ALERT);
        JoinConstraint patientJoin = join("patient");
        patientJoin.add(eq("entity", patient));
        Date now = new Date();

        // duplicate the archetype and date range on the participation to encourage MySQL to use the correct index
        patientJoin.add(new ParticipationConstraint(ActShortName, PatientArchetypes.ALERT));
        patientJoin.add(QueryHelper.createParticipationDateRangeConstraint(now));

        JoinConstraint alertJoin = join("alertType");
        alertJoin.add(join("entity").add(eq("active", true)));
        alertJoin.add(new ParticipationConstraint(ActShortName, PatientArchetypes.ALERT));
        alertJoin.add(QueryHelper.createParticipationDateRangeConstraint(now));

        query.add(patientJoin);
        query.add(alertJoin);
        query.add(QueryHelper.createDateRangeConstraint(now));
        query.add(eq("status", ActStatus.IN_PROGRESS));

        IArchetypeService service = getService();
        IMObjectQueryIterator<Act> iterator = new IMObjectQueryIterator<>(service, query);
        while (iterator.hasNext()) {
            Act act = iterator.next();
            Entity alertType = service.getBean(act).getTarget("alertType", Entity.class);
            if (alertType != null) {
                result.add(new Alert(alertType, act));
            }
        }
        if (result.size() > 1) {
            Collections.sort(result);
        }
        return result;
    }
}
