/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.property;

import org.openvpms.component.business.domain.im.archetype.descriptor.ArchetypeDescriptor;
import org.openvpms.component.business.domain.im.archetype.descriptor.NodeDescriptor;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.system.common.util.Variables;
import org.openvpms.macro.Macros;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.system.ServiceHelper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * A builder for {@link PropertySet} instances.
 * <p/>
 * This allows the default attributes of properties to be overridden.
 *
 * @author Tim Anderson
 */
public class PropertySetBuilder {

    /**
     * The object to build properties for. May be {@code null}
     */
    private final IMObject object;

    /**
     * The properties, keyed on name.
     */
    private final Map<String, Property> properties = new LinkedHashMap<>();

    /**
     * Constructs a {@link PropertySetImpl} from an object.
     *
     * @param object the object
     */
    public PropertySetBuilder(IMObject object) {
        this(object, null);
    }

    /**
     * Constructs a {@link PropertySetImpl} from an object.
     *
     * @param object  the object
     * @param context the layout context. May be {@code null}
     */
    public PropertySetBuilder(IMObject object, LayoutContext context) {
        this(object, getArchetypeDescriptor(object, context), (context != null) ? context.getVariables() : null);
    }

    /**
     * Constructs a {@link PropertySetImpl} for an object and descriptor.
     *
     * @param object    the object
     * @param archetype the archetype descriptor
     * @param variables the variables for macro expansion. May be {@code null}
     */
    public PropertySetBuilder(IMObject object, ArchetypeDescriptor archetype, Variables variables) {
        this(object, archetype, variables, ServiceHelper.getMacros());
    }

    /**
     * Constructs a {@link PropertySetImpl} for an object and descriptor.
     *
     * @param object    the object
     * @param archetype the archetype descriptor
     * @param variables the variables for macro expansion. May be {@code null}
     * @param macros    the macros
     */
    public PropertySetBuilder(IMObject object, ArchetypeDescriptor archetype, Variables variables, Macros macros) {
        this.object = object;

        if (archetype == null) {
            throw new IllegalStateException(
                    "No archetype descriptor for object, id=" + object.getId() + ", archetype="
                    + object.getArchetype());
        }

        for (NodeDescriptor descriptor : archetype.getAllNodeDescriptors()) {
            Property property = new IMObjectProperty(object, descriptor);
            // for editable string properties, register a transformer that supports macro expansion with variables
            if (property.isString() && !property.isDerived() && !property.isReadOnly()) {
                property.setTransformer(new StringPropertyTransformer(property, true, macros, object, variables));
            }
            properties.put(descriptor.getName(), property);
        }
    }

    /**
     * Marks a property as required.
     *
     * @param name the property name
     * @return this builder
     */
    public PropertySetBuilder setRequired(String name) {
        Property property = getProperty(name);
        properties.put(name, new RequiredProperty(property));
        return this;
    }

    /**
     * Marks a property as read-only.
     *
     * @param name the property name
     * @return this builder
     */
    public PropertySetBuilder setReadOnly(String name) {
        return setReadOnly(name, true);
    }

    /**
     * Marks a property as read-only or editable.
     *
     * @param name     the property name
     * @param readOnly if {@code true}, the property is read-only, otherwise it is editable
     * @return this builder
     */
    public PropertySetBuilder setReadOnly(String name, boolean readOnly) {
        Property property = getProperty(name);
        properties.put(name, new DelegatingProperty(property) {
            @Override
            public boolean isReadOnly() {
                return readOnly;
            }
        });
        return this;
    }

    /**
     * Marks a property as hidden.
     *
     * @param name the property name
     * @return this builder
     */
    public PropertySetBuilder setHidden(String name) {
        return setHidden(name, true);
    }

    /**
     * Marks a property as hidden or visible.
     *
     * @param name   the property name
     * @param hidden if {@code true}, the property is hidden, otherwise it is visible
     * @return this builder
     */
    public PropertySetBuilder setHidden(String name, boolean hidden) {
        Property property = getProperty(name);
        properties.put(name, new DelegatingProperty(property) {
            @Override
            public boolean isHidden() {
                return hidden;
            }
        });
        return this;
    }

    /**
     * Marks a property as editable.
     *
     * @param name the property name
     * @return this builder
     */
    public PropertySetBuilder setEditable(String name) {
        Property property = getProperty(name);
        properties.put(name, new DelegatingProperty(property) {
            @Override
            public boolean isReadOnly() {
                return false;
            }

            @Override
            public boolean isHidden() {
                return false;
            }
        });
        return this;
    }

    /**
     * Makes a property mutable.
     *
     * @param name the property name
     * @return this
     */
    public PropertySetBuilder mutable(String name) {
        Property property = getProperty(name);
        if (!(property instanceof MutableProperty)) {
            properties.put(name, new MutableProperty(property));
        }
        return this;
    }

    /**
     * Restricts the archetypes that a property supports.
     *
     * @param name       the property name
     * @param archetypes the archetypes to remove
     * @return this
     */
    public PropertySetBuilder removeArchetypesFromRange(String name, String... archetypes) {
        Property property = getProperty(name);
        properties.put(name, new RestrictedArchetypeProperty(property, archetypes));
        return this;
    }

    /**
     * Builds the property set.
     *
     * @return the new property set
     */
    public PropertySet build() {
        return new PropertySetImpl(object, properties.values());
    }

    /**
     * Returns the named property.
     *
     * @param name       the property name
     * @return the corresponding property
     * @throws IllegalArgumentException if the property is not found
     */
    protected Property getProperty(String name) {
        Property property = properties.get(name);
        if (property == null) {
            throw new IllegalArgumentException("Argument 'name' doesnt refer to a valid property: " + name);
        }
        return property;
    }

    /**
     * Returns the archetype descriptor for an object.
     *
     * @param object  the object
     * @param context the layout context. May be {@code null}
     * @return the archetype descriptor for the object
     */
    private static ArchetypeDescriptor getArchetypeDescriptor(IMObject object, LayoutContext context) {
        return (context != null) ? context.getArchetypeDescriptor(object)
                                 : DescriptorHelper.getArchetypeDescriptor(object, ServiceHelper.getArchetypeService());
    }

    /**
     * Restricts the archetypes that a property supports.
     */
    private static class RestrictedArchetypeProperty extends DelegatingProperty {

        /**
         * The archetypes to remove.
         */
        private final List<String> archetypesToRemove;

        /**
         * Constructs a {@code RestrictedArchetypeProperty}
         *
         * @param property           the property to delegate to
         * @param archetypesToRemove the archetypes to remove
         */
        public RestrictedArchetypeProperty(Property property, String[] archetypesToRemove) {
            super(property);
            this.archetypesToRemove = Arrays.asList(archetypesToRemove);
        }

        /**
         * Returns the archetype short names that this property may support.
         *
         * @return the archetype short names
         */
        @Override
        public String[] getArchetypeRange() {
            String[] archetypeRange = super.getArchetypeRange();
            List<String> result = new ArrayList<>(Arrays.asList(archetypeRange));
            return result.removeAll(archetypesToRemove) ? result.toArray(new String[0]) : archetypeRange;
        }
    }
}
