/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.mail;

import org.openvpms.component.model.party.Contact;
import org.openvpms.web.component.bound.BoundTextComponentFactory;
import org.openvpms.web.component.property.ModifiableListener;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.Validator;

import java.util.List;

/**
 * Binds a property to an email address selector, with an editable text field.
 * Addresses can be selected, or entered manually. Entered addresses don't need to appear in the list.
 *
 * @author Tim Anderson
 */
public class EditableAddressSelector extends AddressSelector {

    /**
     * The available email contacts.
     */
    private final List<Contact> contacts;

    /**
     * The property.
     */
    private final Property property;

    /**
     * The property listener.
     */
    private final ModifiableListener propertyListener;

    /**
     * The selected contact.
     */
    private Contact selected;

    /**
     * Constructs an {@link EditableAddressSelector}.
     *  @param property  the property
     * @param contacts  the available contacts
     */
    public EditableAddressSelector(Property property, List<Contact> contacts) {
        super(contacts, new FromAddressFormatter(), BoundTextComponentFactory.create(property, 40));
        this.contacts = contacts;
        this.property = property;
        propertyListener = modifiable -> updateSelectionFromProperty();
        updateSelectionFromProperty();
    }

    /**
     * Sets the selected contact.
     *
     * @param contact the contact. May be {@code null}
     */
    @Override
    public void setSelected(Contact contact) {
        setSelected(contact, true);
    }

    /**
     * Returns the selected contact.
     *
     * @return the selected contact. May be {@code null} if there are no contacts, or the entered email doesn't match
     * one of those listed
     */
    @Override
    public Contact getSelected() {
        return selected;
    }

    /**
     * Determines if the object has been modified.
     *
     * @return {@code true} if the object has been modified
     */
    @Override
    public boolean isModified() {
        return super.isModified() || property.isModified();
    }

    /**
     * Clears the modified status of the object.
     */
    @Override
    public void clearModified() {
        super.clearModified();
        property.clearModified();
    }

    /**
     * Validates the object.
     *
     * @param validator the validator
     * @return {@code true} if the object and its descendants are valid otherwise {@code false}
     */
    @Override
    protected boolean doValidation(Validator validator) {
        return property.validate(validator);
    }

    /**
     * Sets the selected contact.
     *
     * @param contact        the contact. May be {@code null}
     * @param updateProperty if {@code true}, update the property from the contact
     */
    private void setSelected(Contact contact, boolean updateProperty) {
        super.setSelected(contact);
        this.selected = contact;
        if (updateProperty) {
            property.removeModifiableListener(propertyListener);
            try {
                String address = (contact != null) ? getFormatter().getAddress(contact) : null;
                property.setValue(address);
            } finally {
                property.addModifiableListener(propertyListener);
            }
        }
    }

    /**
     * Update the selected contact from the property.
     */
    private void updateSelectionFromProperty() {
        String address = property.getString();
        Contact contact = address != null ? AddressSelector.getContact(contacts, address) : null;
        setSelected(contact, false);
    }
}