/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.query;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.component.business.service.archetype.ArchetypeServiceHelper;
import org.openvpms.component.business.service.archetype.handler.ArchetypeHandler;
import org.openvpms.component.business.service.archetype.handler.ArchetypeHandlers;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.system.common.query.ArchetypeQueryException;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.system.ServiceHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * A factory for {@link Query} instances. The factory is configured to return
 * specific {@link Query} implementations based on the supplied criteria, with
 * {@link DefaultQuery} returned if no implementation matches.
 * <p>
 * The factory is configured using a <em>QueryFactory.properties</em> file,
 * located in the class path. The file contains pairs of archetype short names
 * and their corresponding query implementations. Short names may be wildcarded
 * e.g:
 * <p>
 * <table> <tr><td>classification.*</td><td>org.openvpms.web.component.im.query.AutoQuery</td></tr>
 * <tr><td>lookup.*</td><td>org.openvpms.web.component.im.query.AutoQuery</td></tr>
 * <tr><td>party.patient*</td><td>org.openvpms.web.component.im.query.PatientQuery</td></tr>
 * <tr><td>party.organisation*</td>org.openvpms.web.component.im.query.AutoQuery</td></tr>
 * <tr><td>party.supplier*</td>org.openvpms.web.component.im.query.AutoQuery</td></tr>
 * </table>
 * <p>
 * Multiple <em>QueryFactory.properties</em> may be used.
 * <p>
 * Default implementations can be registered in a <em>DefaultQueryFactory.properties</em> file; these are overridden by
 * <em>QueryFactory.properties</em>.
 *
 * @author Tim Anderson
 */
public final class QueryFactory {

    /**
     * Query implementations.
     */
    private static ArchetypeHandlers<Query<?>> queries;

    /**
     * The logger.
     */
    private static final Logger log = LoggerFactory.getLogger(QueryFactory.class);


    /**
     * Prevent construction.
     */
    private QueryFactory() {
    }

    /**
     * Construct a new {@link Query}. Query implementations must provide at
     * least one constructor accepting the following arguments, invoked in the
     * order:
     * <ul>
     * <li>(String[] shortNames, Context context)</li>
     * <li>(String[] shortNames)</li>
     * <li>default constructor</li>
     * </ul>
     *
     * @param shortName the archetype short name to query on. May contain
     *                  wildcards
     * @param context   the context
     * @return a new query
     * @throws ArchetypeQueryException if the short names don't match any
     *                                 archetypes
     */
    @SuppressWarnings("unchecked")
    public static <T extends IMObject> Query<T> create(String shortName, Context context) {
        return (Query<T>) create(new String[]{shortName}, context, IMObject.class);
    }

    /**
     * Construct a new {@link Query}. Query implementations must provide at
     * least one constructor accepting the following arguments, invoked in the
     * order:
     * <ul>
     * <li>(String[] shortNames, Context context)</li>
     * <li>(String[] shortNames)</li>
     * <li>default constructor</li>
     * </ul>
     *
     * @param shortName the archetype short name to query on. May contain wildcards
     * @param context   the context
     * @param type      the type that the query returns
     * @return a new query
     * @throws ArchetypeQueryException if the short names don't match any archetypes
     */
    public static <T> Query<T> create(String shortName, Context context, Class<T> type) {
        return create(new String[]{shortName}, context, type);
    }

    /**
     * Construct a new {@link Query}. Query implementations must provide at least constructor one accepting the
     * following arguments, invoked in the
     * order:
     * <ul>
     * <li>(String[] shortNames, Context context)</li>
     * <li>(String[] shortNames)</li>
     * <li>default constructor</li>
     * </ul>
     *
     * @param shortNames the archetype short names to query on. May contain wildcards
     * @param context    the current context
     * @return a new query
     * @throws ArchetypeQueryException if the short names don't match any
     *                                 archetypes
     */
    @SuppressWarnings("unchecked")
    public static <T extends IMObject> Query<T> create(String[] shortNames, Context context) {
        return (Query<T>) create(shortNames, context, IMObject.class);
    }

    /**
     * Construct a new {@link Query}. Query implementations must provide at least constructor one accepting the
     * following arguments, invoked in the order:
     * <ul>
     * <li>(String[] shortNames, Context context)</li>
     * <li>(String[] shortNames)</li>
     * <li>default constructor</li>
     * </ul>
     *
     * @param shortNames the archetype short names to query on. May contain wildcards
     * @param context    the current context
     * @param type       the type that the query returns
     * @return a new query
     * @throws ArchetypeQueryException if the short names don't match any archetypes
     * @throws QueryException          if no query supports the supplied short names and type
     */
    public static <T> Query<T> create(String[] shortNames, Context context, Class<T> type) {
        return create(shortNames, true, context, type);
    }


    /**
     * Construct a new {@link Query}. Query implementations must provide at least constructor one accepting the
     * following arguments, invoked in the order:
     * <ul>
     * <li>(String[] shortNames, Context context)</li>
     * <li>(String[] shortNames)</li>
     * <li>default constructor</li>
     * </ul>
     *
     * @param shortNames the archetype short names to query on. May contain wildcards
     * @param exact      if {@code true}, all short names must have the same handler and configuration.
     *                   If {@code false}, all short names must match a common handler
     * @param context    the current context
     * @param type       the type that the query returns
     * @return a new query
     * @throws ArchetypeQueryException if the short names don't match any archetypes
     * @throws QueryException          if no query supports the supplied short names and type
     */
    public static <T> Query<T> create(String[] shortNames, boolean exact, Context context, Class<T> type) {
        Query<T> query;
        shortNames = DescriptorHelper.getShortNames(shortNames, ServiceHelper.getArchetypeService());
        ArchetypeHandler<Query<?>> handler = getQueries().getHandler(shortNames, exact);
        if (handler == null) {
            query = createDefaultQuery(shortNames, type);
        } else {
            query = create(handler, shortNames, context, type);
        }
        if (query == null) {
            throw new QueryException(QueryException.ErrorCode.NoQuery, StringUtils.join(shortNames, ", "),
                                     type.getName());
        }
        return query;
    }

    /**
     * Initialise a query.
     *
     * @param query the query to initialise
     */
    @SuppressWarnings("unchecked")
    public static <T> void initialise(Query<T> query) {
        Class<Query<T>> type = (Class<Query<T>>) query.getClass();
        ArchetypeHandlers<Query<?>> handlers = getQueries();
        ArchetypeHandler<Query<?>> handler = handlers.getHandler(type);
        if (handler != null) {
            initialise(query, handler);
        } else {
            handler = handlers.getHandler(query.getShortNames());
            if (handler != null && handler.getType().isAssignableFrom(type)) {
                initialise(query, handler);
            }
        }
    }

    /**
     * Initialise a query.
     *
     * @param query   the query
     * @param handler the query handler
     */
    protected static void initialise(Query<?> query, ArchetypeHandler<Query<?>> handler) {
        try {
            handler.initialise(query);
        } catch (Throwable exception) {
            log.error(exception.getMessage(), exception);
        }
    }

    /**
     * Attempts to create a new query, using one of the following constructors:
     * <ul>
     * <li>(String[] shortNames, Context context)</li>
     * <li>(String[] shortNames)</li>
     * <li>(Context)</li>
     * <li>default constructor</li>
     * </ul>
     *
     * @param handler    the {@link Query} implementation
     * @param shortNames the archetype short names to query on
     * @param context    the context
     * @param type       the type that the query returns
     * @return a new query, or {@code null} if no appropriate constructor can be found or construction fails
     */
    @SuppressWarnings("unchecked")
    private static <T> Query<T> create(ArchetypeHandler<Query<?>> handler, String[] shortNames, Context context,
                                       Class<T> type) {
        Query<?> result = null;
        try {
            try {
                Object[] args = new Object[]{shortNames, context};
                result = handler.create(args);
            } catch (NoSuchMethodException exception) {
                try {
                    Object[] args = new Object[]{shortNames};
                    result = handler.create(args);
                } catch (NoSuchMethodException nested) {
                    try {
                        Object[] args = new Object[]{context};
                        result = handler.create(args);
                    } catch (NoSuchMethodException nested2) {
                        result = handler.create();
                    }
                }
            }
            if (!type.isAssignableFrom(result.getType())) {
                result = null;
            }
        } catch (Throwable throwable) {
            log.error(throwable.getMessage(), throwable);
        }
        return (Query<T>) result;
    }

    /**
     * Creates a new default query for the supplied archetype short names and type.
     *
     * @param shortNames the archetype short names to query
     * @param type       the type that the query returns
     * @return a new query, or {@code null} if there is no default query for the specified type
     */
    @SuppressWarnings("unchecked")
    private static <T> Query<T> createDefaultQuery(String[] shortNames, Class<?> type) {
        if (IMObject.class.isAssignableFrom(type)) {
            return (Query<T>) new DefaultQuery<>(shortNames, (Class<IMObject>) type);
        }
        return null;
    }

    /**
     * Returns the query implementations.
     *
     * @return the queries
     */
    @SuppressWarnings("unchecked")
    private static synchronized ArchetypeHandlers<Query<?>> getQueries() {
        if (queries == null) {
            Class<?> type = Query.class;
            queries = new ArchetypeHandlers<>("QueryFactory.properties", "DefaultQueryFactory.properties",
                                              (Class<Query<?>>) type, "query.",
                                              ArchetypeServiceHelper.getArchetypeService());
        }
        return queries;
    }

}
