/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2020 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.query;

import echopointng.DateField;
import nextapp.echo2.app.Component;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.web.component.bound.BoundCheckBox;
import org.openvpms.web.component.bound.BoundDateFieldFactory;
import org.openvpms.web.component.im.view.ComponentState;
import org.openvpms.web.component.property.ModifiableListener;
import org.openvpms.web.component.property.ModifiableListeners;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.SimpleProperty;
import org.openvpms.web.component.util.ComponentHelper;
import org.openvpms.web.echo.factory.RowFactory;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.echo.style.Styles;
import org.openvpms.web.resource.i18n.Messages;

import java.util.Date;


/**
 * Component for selecting a range of dates.
 *
 * @author Tim Anderson
 */
public class DateRange {

    /**
     * The focus group.
     */
    private final FocusGroup focus;

    /**
     * Determines if the 'allDates' check box should be displayed.
     */
    private final boolean showAll;

    /**
     * Indicates if all dates should be selected. If so, the from and to dates are ignored.
     */
    private final SimpleProperty all = new SimpleProperty("all", true, Boolean.class, Messages.get("daterange.all"));

    /**
     * The from date.
     */
    private final SimpleProperty from = new SimpleProperty("from", null, Date.class, Messages.get("daterange.from"));

    /**
     * The to date.
     */
    private final SimpleProperty to = new SimpleProperty("to", null, Date.class, Messages.get("daterange.to"));

    /**
     * The from property listener.
     */
    private final ModifiableListener fromListener;

    /**
     * The to property listener.
     */
    private final ModifiableListener toListener;

    /**
     * The listeners.
     */
    private final ModifiableListeners listeners = new ModifiableListeners();

    /**
     * The all-dates component.
     */
    private ComponentState allDates;

    /**
     * The from-date component.
     */
    private ComponentState fromDate;

    /**
     * The to-date component.
     */
    private ComponentState toDate;

    /**
     * The component.
     */
    private Component component;

    /**
     * Constructs a {@link DateRange}.
     */
    public DateRange() {
        this(true);
    }

    /**
     * Constructs a {@link DateRange}.
     *
     * @param showAll determines if an 'all' checkbox should be displayed to select all dates
     */
    public DateRange(boolean showAll) {
        this(showAll, DateRules.getToday());
    }

    /**
     * Constructs a {@link DateRange}.
     *
     * @param showAll determines if an 'all' checkbox should be displayed to select all dates
     * @param date    the date to assign the 'from' and 'to' dates. May  be {@code null}
     */
    public DateRange(boolean showAll, Date date) {
        this(showAll, date, date);
    }

    /**
     * Constructs a {@link DateRange}.
     *
     * @param showAll determines if an 'all' checkbox should be displayed to select all dates
     * @param from    the 'from' date. May be {@code null}
     * @param to      the 'to' date. May be {@code null}
     */
    public DateRange(boolean showAll, Date from, Date to) {
        this.showAll = showAll;
        focus = new FocusGroup("DateRange");
        if (showAll) {
            all.addModifiableListener(modifiable -> onAllDatesChanged());
        }
        fromListener = modifiable -> onFromChanged();
        toListener = modifiable -> onToChanged();
        setFrom(from);
        setTo(to);
        this.from.addModifiableListener(fromListener);
        this.to.addModifiableListener(toListener);
    }

    /**
     * Returns the 'from' date.
     *
     * @return the 'from' date, or {@code null} to query all dates
     */
    public Date getFrom() {
        return getAllDates() ? null : getDate(from);
    }

    /**
     * Sets the 'from' date.
     *
     * @param date the 'from' date
     */
    public void setFrom(Date date) {
        setFrom(date, false);
    }

    /**
     * Returns the 'to' date.
     *
     * @return the 'to' date, or {@code null} to query all dates
     */
    public Date getTo() {
        return getAllDates() ? null : getDate(to);
    }

    /**
     * Sets the 'to' date.
     *
     * @param date the 'to' date
     */
    public void setTo(Date date) {
        to.setValue(date);
    }

    /**
     * Sets the state of the <em>allDates</em> checkbox, if present.
     *
     * @param selected the state of the <em>allDates</em> checkbox
     */
    public void setAllDates(boolean selected) {
        if (showAll) {
            all.setValue(selected);
        }
    }

    /**
     * Sets the enabled state of the date range.
     *
     * @param enabled if {@code true} enable the component otherwise disable it
     */
    public void setEnabled(boolean enabled) {
        if (allDates != null) {
            allDates.getComponent().setEnabled(enabled);
        }
        if (enabled && !getAllDates()) {
            setDateFieldsEnabled(true);
        } else {
            setDateFieldsEnabled(false);
        }
    }

    /**
     * Determines if all dates are being selected.
     *
     * @return {@code true} if all dates are being selected
     */
    public boolean getAllDates() {
        return showAll && all.getValue() != null && (Boolean) all.getValue();
    }

    /**
     * Renders the component.
     *
     * @return the component
     */
    public Component getComponent() {
        if (component == null) {
            component = doLayout(getContainer());
        }
        return component;
    }

    /**
     * Lays out the component in the specified container
     *
     * @param container the container
     */
    public void setContainer(Component container) {
        component = doLayout(container);
    }

    /**
     * Returns the focus group.
     *
     * @return the focus group.
     */
    public FocusGroup getFocusGroup() {
        return focus;
    }

    /**
     * Adds a listener to be notified when this changes.
     *
     * @param listener the listener to add
     */
    public void addListener(ModifiableListener listener) {
        listeners.addListener(listener);
    }

    /**
     * Removes a listener.
     *
     * @param listener the listener to remove
     */
    public void removeListener(ModifiableListener listener) {
        listeners.removeListener(listener);
    }

    /**
     * Sets the 'from' date.
     *
     * @param date    the 'from' date
     * @param disable if {@code true} disable listeners
     */
    protected void setFrom(Date date, boolean disable) {
        if (disable) {
            from.removeModifiableListener(fromListener);
        }
        from.setValue(date);
        if (disable) {
            from.addModifiableListener(fromListener);
        }
    }

    /**
     * Sets the 'to' date.
     *
     * @param date    the 'to' date
     * @param disable if {@code true} disable listeners
     */
    protected void setTo(Date date, boolean disable) {
        if (disable) {
            to.removeModifiableListener(toListener);
        }
        to.setValue(date);
        if (disable) {
            to.addModifiableListener(toListener);
        }
    }

    /**
     * Lays out the component.
     *
     * @param container the container
     * @return the component
     */
    protected Component doLayout(Component container) {
        fromDate = createFromDate(from);
        toDate = createToDate(to);
        if (showAll) {
            allDates = createAllDates(all);
        } else {
            allDates = null;
        }
        if (allDates != null) {
            container.add(allDates.getLabel());
            container.add(allDates.getComponent());
            focus.add(allDates.getComponent());
        }
        container.add(fromDate.getLabel());
        container.add(fromDate.getComponent());
        container.add(toDate.getLabel());
        container.add(toDate.getComponent());

        setDateFieldsEnabled(!getAllDates());

        focus.add(fromDate.getComponent());
        focus.add(toDate.getComponent());
        return container;
    }

    /**
     * Returns a container to render the component.
     *
     * @return the container
     */
    protected Component getContainer() {
        return RowFactory.create(Styles.CELL_SPACING);
    }

    /**
     * Creates a component to render the "all dates" property.
     *
     * @param allDates the "all dates" property
     * @return a new component
     */
    protected ComponentState createAllDates(Property allDates) {
        return new ComponentState(new BoundCheckBox(allDates), allDates);
    }

    /**
     * Creates a component to render the "from date" property.
     *
     * @param from the "from date" property
     * @return a new component
     */
    protected ComponentState createFromDate(Property from) {
        return new ComponentState(BoundDateFieldFactory.create(from), from);
    }

    /**
     * Creates a component to render the "to date" property.
     *
     * @param to the "to date" property
     * @return a new component
     */
    protected ComponentState createToDate(Property to) {
        return new ComponentState(BoundDateFieldFactory.create(to), to);
    }

    /**
     * Invoked when the 'all dates' check box changes.
     */
    protected void onAllDatesChanged() {
        boolean enabled = !getAllDates();
        setDateFieldsEnabled(enabled);
        listeners.notifyListeners(all);
    }

    /**
     * Updates the 'from' date when the 'to' date changes.
     * <p/>
     * This implementation sets the 'from' date = 'to' if 'from' is greater.
     *
     * @param from the 'from' date. May be {@code null}
     * @param to   the 'to' date. May be {@code null}
     */
    protected void updateFromDate(Date from, Date to) {
        if (from != null && to != null && DateRules.compareDates(from, to) > 0) {
            setFrom(to, true);
        }
    }

    /**
     * Updates the 'to' date when the from date changes.
     * <p/>
     * This implementation sets the 'to' date = 'from' if 'from' is greater.
     *
     * @param from the 'from' date. May be {@code null}
     * @param to   the 'to' date. May be {@code null}
     */
    protected void updateToDate(Date from, Date to) {
        if (from != null && to != null && DateRules.compareDates(from, to) > 0) {
            setTo(from, true);
        }
    }

    /**
     * Invoked when the 'from' date changes.
     * <p/>
     * Delegates to {@link #updateToDate(Date, Date)} and notifies listeners.
     */
    private void onFromChanged() {
        updateToDate(getFrom(), getTo());
        listeners.notifyListeners(all);
    }

    /**
     * Invoked when the 'to' date changes. Sets the 'from' date = 'to' if 'from' is greater.
     * <p/>
     * Delegates to {@link #updateFromDate(Date, Date)} and notifies listeners.
     */
    private void onToChanged() {
        updateFromDate(getFrom(), getTo());
        listeners.notifyListeners(to);
    }

    /**
     * Enables/disables the date fields.
     *
     * @param enabled if {@code true}, enable them, else disable them
     */
    private void setDateFieldsEnabled(boolean enabled) {
        ComponentHelper.enable(fromDate.getLabel(), enabled);
        ComponentHelper.enable((DateField) fromDate.getComponent(), enabled);
        ComponentHelper.enable(toDate.getLabel(), enabled);
        ComponentHelper.enable((DateField) toDate.getComponent(), enabled);
    }

    /**
     * Returns the date of the given field.
     *
     * @param property the date property
     * @return the selected date
     */
    private Date getDate(SimpleProperty property) {
        Date date = (Date) property.getValue();
        return DateRules.getDate(date); // truncate any time component
    }

}
