/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.edit.payment;

import org.openvpms.archetype.rules.supplier.account.SupplierAccountRules;
import org.openvpms.component.business.domain.im.datatypes.quantity.Money;
import org.openvpms.component.business.service.archetype.helper.TypeHelper;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.party.Party;
import org.openvpms.web.component.im.act.ActHelper;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.util.IMObjectCreationListener;
import org.openvpms.web.system.ServiceHelper;

import java.math.BigDecimal;


/**
 * An editor for {@link Act}s which have an archetype of
 * <em>act.supplierAccountPayment</em> or <em>act.supplierAccountRefund</em>.
 *
 * @author Tim Anderson
 */
public class SupplierPaymentEditor extends PaymentEditor {

    /**
     * Constructs a {@link SupplierPaymentEditor}.
     *
     * @param act     the act to edit
     * @param parent  the parent object. May be {@code null}
     * @param context the layout context
     */
    public SupplierPaymentEditor(FinancialAct act, IMObject parent, LayoutContext context) {
        super(act, parent, context);
        initParticipant("supplier", context.getContext().getSupplier());
        getItems().setCreationListener(new IMObjectCreationListener() {
            public void created(IMObject object) {
                onCreated((FinancialAct) object);
            }
        });
    }

    /**
     * Invoked when a child act is created.
     * If the act is a payment, defaults the total to the outstanding balance.
     *
     * @param act the act
     */
    private void onCreated(FinancialAct act) {
        if (TypeHelper.isA(act, "act.supplierAccountPayment*")) {
            // Default the amount to the outstanding balance
            Party supplier = (Party) getParticipant("supplier");
            if (supplier != null) {
                FinancialAct parent = getObject();
                BigDecimal diff = ActHelper.sum(parent, "amount");

                SupplierAccountRules rules = ServiceHelper.getBean(SupplierAccountRules.class);
                BigDecimal current = rules.getBalance(supplier);

                BigDecimal balance = current.subtract(diff);
                act.setTotal(new Money(balance));
                getItems().setModified(act, true);
            }
        }
    }

}
