/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.edit;

import echopointng.KeyStrokes;
import nextapp.echo2.app.Button;
import nextapp.echo2.app.Component;
import nextapp.echo2.app.SplitPane;
import nextapp.echo2.app.button.AbstractButton;
import nextapp.echo2.app.event.ActionEvent;
import org.apache.commons.lang3.StringUtils;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.edit.AlertListener;
import org.openvpms.web.component.im.view.Selection;
import org.openvpms.web.component.macro.MacroDialog;
import org.openvpms.web.component.property.DefaultValidator;
import org.openvpms.web.component.property.ValidationHelper;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.echo.button.ButtonSet;
import org.openvpms.web.echo.dialog.PopupDialog;
import org.openvpms.web.echo.error.ErrorHandler;
import org.openvpms.web.echo.event.ActionListener;
import org.openvpms.web.echo.event.Vetoable;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.resource.i18n.Messages;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.function.Consumer;


/**
 * A popup dialog that displays an {@link IMObjectEditor}.
 *
 * @author Tim Anderson
 */
public abstract class AbstractEditDialog extends PopupDialog {

    /**
     * Edit dialog style name.
     */
    protected static final String STYLE = "EditDialog";

    /**
     * The alert manager.
     */
    private final AlertManager alerts;

    /**
     * Determines if the dialog should save when apply and OK are pressed.
     */
    private final boolean userSave;

    /**
     * Determines if the object can be saved.
     */
    private final boolean save;

    /**
     * The context.
     */
    private final Context context;

    /**
     * The editor.
     */
    private IMObjectEditor editor;

    /**
     * Determines if saves are disabled.
     */
    private boolean savedDisabled;

    /**
     * The current component.
     */
    private Component current;

    /**
     * The last archetype edited. Used to determine if resizing is required.
     */
    private String lastArchetype;

    /**
     * The current component focus group.
     */
    private FocusGroup currentGroup;

    /**
     * The current help context.
     */
    private HelpContext helpContext;

    /**
     * Callback to be invoked when the editor is saved, within the same transaction.
     */
    private Consumer<IMObjectEditor> transactionCallback;

    /**
     * Callback to be invoked after the editor is successfully saved.
     */
    private Consumer<IMObjectEditor> postSaveCallback;

    /**
     * The logger.
     */
    private static final Logger log = LoggerFactory.getLogger(AbstractEditDialog.class);

    /**
     * Constructs an {@link AbstractEditDialog}.
     *
     * @param title   the dialog title
     * @param actions the edit actions
     * @param context the context
     * @param help    the help context
     */
    public AbstractEditDialog(String title, EditActions actions, Context context, HelpContext help) {
        this(null, title, actions, context, help);
    }

    /**
     * Constructs an {@link AbstractEditDialog}.
     *
     * @param editor  the editor
     * @param actions the edit actions
     * @param context the context
     */
    public AbstractEditDialog(IMObjectEditor editor, EditActions actions, Context context) {
        this(editor, editor.getTitle(), actions, context, editor.getHelpContext());
    }

    /**
     * Constructs an {@link AbstractEditDialog}.
     *
     * @param editor  the editor. May be {@code null}
     * @param title   the dialog title. May be {@code null}
     * @param actions the edit actions
     * @param context the context
     * @param help    the help context. May be {@code null}
     */
    protected AbstractEditDialog(IMObjectEditor editor, String title, EditActions actions, Context context,
                                 HelpContext help) {
        super(title, STYLE, actions.getButtons(), help);
        this.context = context;
        save = actions.save();
        userSave = save && actions.userSave();
        alerts = new AlertManager(getContentPane(), 2);
        setModal(true);
        setEditor(editor);
        getButtons().addKeyListener(KeyStrokes.ALT_MASK | KeyStrokes.VK_M, new ActionListener() {
            public void onAction(ActionEvent event) {
                onMacro();
            }
        });
        setCancelListener(this::onCancel);
    }

    /**
     * Returns the editor.
     *
     * @return the editor, or {@code null} if none has been set
     */
    public IMObjectEditor getEditor() {
        return editor;
    }

    /**
     * Registers a callback to be invoked whenever the editor is saved.
     * <p>
     * This is invoked within the same transaction as the editor. If it throws an exception, any changes will
     * be rolled back.
     *
     * @param callback the callback. May be {@code null}
     */
    public void setTransactionCallback(Consumer<IMObjectEditor> callback) {
        this.transactionCallback = callback;
    }

    /**
     * Registers a callback to be invoked each time the editor is successfully saved.
     * <p>
     * This is invoked after the save transaction has completed.
     *
     * @param callback the callback. May be {@code null}
     */
    public void setPostSaveCallback(Consumer<IMObjectEditor> callback) {
        postSaveCallback = callback;
    }

    /**
     * Saves the current object, if saving is enabled.
     * <p>
     * If it is, and the object is valid, then {@link #doSave(IMObjectEditor)} is called.
     * If {@link #doSave(IMObjectEditor)} fails (i.e. returns {@code false}), then {@link #saveFailed()} is called.
     *
     * @return {@code true} if the object was saved
     */
    public boolean save() {
        boolean result = false;
        if (canSave()) {
            // save the editor in a transaction, reloading it if the save fails.
            IMObjectEditorSaver saver = new IMObjectEditorSaver(new ReloadingSaveOperation());
            result = saver.save(editor);
            if (result) {
                result = postSave();
            }
        }
        return result;
    }

    /**
     * Saves the editor, optionally closing the dialog.
     * <p>
     * If the save fails, the dialog will remain open.
     *
     * @param close if {@code true} close the dialog
     */
    public void save(boolean close) {
        if (!close) {
            onApply();
        } else {
            onOK();
        }
    }

    /**
     * Determines if a skip button should be added.
     *
     * @param skip if {@code true} add a skip button, otherwise remove it
     */
    public void addSkip(boolean skip) {
        ButtonSet buttons = getButtons();
        AbstractButton button = buttons.getButton(SKIP_ID);
        if (skip) {
            if (button == null) {
                addButton(SKIP_ID, false);
            }
        } else {
            if (button != null) {
                buttons.remove(button);
            }
        }
    }

    /**
     * Returns the help context.
     *
     * @return the help context
     */
    @Override
    public HelpContext getHelpContext() {
        return (helpContext != null) ? helpContext : super.getHelpContext();
    }

    /**
     * Sets the action and closes the window.
     *
     * @param action the action
     */
    @Override
    public void close(String action) {
        alerts.clear();
        super.close(action);
    }

    /**
     * Resizes the dialog if required.
     * <p/>
     * This looks for a style sheet property name {@code <styleName>.size.<archetype>} with a value of the format
     * {@code <width>x<height>}<br/>
     * The width and height are expressed in terms of the font units that will be converted to pixels using the font
     * size.
     */
    @Override
    protected void resize() {
        IMObjectEditor editor = getEditor();
        if (editor != null) {
            String archetype = editor.getObject().getArchetype();
            if (!StringUtils.equals(archetype, lastArchetype)) {
                String property = getStyleName() + ".size." + archetype;
                if (resize(property)) {
                    lastArchetype = archetype;
                }
            }
        }
    }

    /**
     * Lays out the component prior to display.
     */
    @Override
    protected void doLayout() {
        super.doLayout();
        if (editor != null) {
            FocusGroup group = editor.getFocusGroup();
            if (group != null) {
                group.setFocus();
            }
        }
    }

    /**
     * Determines if the current object can be saved.
     *
     * @return {@code true} if the current object can be saved
     */
    protected boolean canSave() {
        return !savedDisabled && save && editor != null;
    }

    /**
     * Saves the current object, if saving is enabled.
     */
    @Override
    protected void onApply() {
        if (userSave) {
            save();
        }
    }

    /**
     * Saves the current object, if saving is enabled, and closes the editor.
     */
    @Override
    protected void onOK() {
        if (userSave) {
            if (save()) {
                close(OK_ID);
            }
        } else if (editor != null) {
            // only close the editor if the object is valid, and no new alerts are displayed
            Validator validator = new DefaultValidator();
            int newAlerts = alerts.getAlertCount();
            if (!editor.validate(validator)) {
                ValidationHelper.showError(validator);
            } else if (newAlerts == alerts.getAlertCount()) {
                // only close the dialog if no new alerts were created
                close(OK_ID);
            }
        } else {
            close(OK_ID);
        }
    }

    /**
     * Close the editor, discarding any unsaved changes.
     */
    @Override
    protected void doCancel() {
        if (editor != null) {
            editor.cancel();
        }
        super.doCancel();
    }

    /**
     * Sets the editor.
     * <p>
     * If there is an existing editor, its selection path will be set on the editor.
     *
     * @param editor the editor. May be {@code null}
     */
    protected void setEditor(IMObjectEditor editor) {
        IMObjectEditor previous = this.editor;
        List<Selection> path = (editor != null && previous != null) ? previous.getSelectionPath() : null;
        setEditor(editor, path);
    }

    /**
     * Sets the editor.
     *
     * @param editor the editor. May be {@code null}
     * @param path   the selection path. May be {@code null}
     */
    protected void setEditor(IMObjectEditor editor, List<Selection> path) {
        IMObjectEditor previous = this.editor;
        if (editor != null) {
            setTitle(editor.getTitle());
            editor.addPropertyChangeListener(IMObjectEditor.COMPONENT_CHANGED_PROPERTY, event -> onComponentChange());
        }
        this.editor = editor;
        if (previous != null) {
            removeEditor(previous);
        } else {
            path = null;
        }
        if (editor != null) {
            addEditor(editor);
            if (path != null) {
                editor.setSelectionPath(path);
            }
        }
    }

    /**
     * Saves the current object.
     *
     * @param editor the editor
     * @throws OpenVPMSException if the save fails
     */
    protected void doSave(IMObjectEditor editor) {
        editor.save();
        if (transactionCallback != null) {
            transactionCallback.accept(editor);
        }
    }

    /**
     * Performs tasks after {@link #save()} has returned successfully.
     * <p/>
     * This implementation invokes the {@link #setPostSaveCallback(Consumer) postSaveCallback}, if any is registered.
     *
     * @return {@code true} if there is no callback, or the callback completed successfully
     */
    protected boolean postSave() {
        boolean result = false;
        if (postSaveCallback == null) {
            result = true;
        } else {
            try {
                postSaveCallback.accept(editor);
                result = true;
            } catch (Throwable exception) {
                String displayName = editor.getDisplayName();
                String title = Messages.format("imobject.save.failed", displayName);
                ErrorHelper.show(title, displayName, editor.getObject(), exception);
                saveFailed();
            }
        }
        return result;
    }

    /**
     * Invoked to reload the object being edited when save fails.
     *
     * @param editor the editor
     * @return {@code true} if the editor was reloaded
     */
    protected boolean reload(IMObjectEditor editor) {
        IMObjectEditor newEditor = null;
        try {
            newEditor = editor.newInstance();
            if (newEditor != null) {
                if (newEditor.getClass() == editor.getClass()) {
                    setEditor(newEditor);
                } else {
                    log.error("Cannot reload editor. Reloaded editor=" + newEditor.getClass().getName()
                              + " is a different type to the original editor=" + editor.getClass().getName());
                }
            }
        } catch (Throwable exception) {
            log.error("Failed to reload editor", exception);
        }
        return newEditor != null;
    }

    /**
     * Invoked to display a message that saving failed, and the editor has been reverted.
     *
     * @param title     the message title
     * @param message   the message
     * @param oldEditor the previous instance of the editor
     */
    protected void reloaded(String title, String message, IMObjectEditor oldEditor) {
        ErrorHandler.getInstance().error(title, message, null, null);
    }

    /**
     * Invoked by {@link #save} when saving fails.
     * <p>
     * This implementation disables saves.
     * TODO - this is a workaround for OVPMS-855
     */
    protected void saveFailed() {
        savedDisabled = true;
        ButtonSet buttons = getButtons();
        for (Component component : buttons.getContainer().getComponents()) {
            if (component instanceof Button) {
                Button button = (Button) component;
                if (!CANCEL_ID.equals(button.getId())) {
                    buttons.setEnabled(button.getId(), false);
                }
            }
        }
    }

    /**
     * Adds the editor to the layout, setting the focus if the dialog is displayed.
     *
     * @param editor the editor
     */
    protected void addEditor(IMObjectEditor editor) {
        setComponent(editor.getComponent(), editor.getFocusGroup(), editor.getHelpContext());

        // register a listener to handle alerts
        editor.setAlertListener(getAlertListener());
        resize();
    }

    /**
     * Removes the editor from the layout.
     *
     * @param editor the editor to remove
     */
    protected void removeEditor(IMObjectEditor editor) {
        editor.setAlertListener(null);
        removeComponent();
    }

    /**
     * Sets the component.
     *
     * @param component the component
     * @param group     the focus group
     * @param context   the help context
     */
    protected void setComponent(Component component, FocusGroup group, HelpContext context) {
        setComponent(component, group, context, true);
    }

    /**
     * Sets the component.
     *
     * @param component the component
     * @param group     the focus group
     * @param context   the help context
     * @param focus     if {@code true}, move the focus
     */
    protected void setComponent(Component component, FocusGroup group, HelpContext context, boolean focus) {
        SplitPane layout = getLayout();
        if (current != null) {
            layout.remove(current);
        }
        if (currentGroup != null) {
            getFocusGroup().remove(currentGroup);
        }
        layout.add(component);
        getFocusGroup().add(0, group);
        if (focus && getParent() != null) {
            // focus in the component
            group.setFocus();
        }
        current = component;
        currentGroup = group;
        helpContext = context;
    }

    /**
     * Removes the existing component and any alerts.
     */
    protected void removeComponent() {
        if (current != null) {
            getLayout().remove(current);
            current = null;
        }
        alerts.clear();
        if (currentGroup != null) {
            getFocusGroup().remove(currentGroup);
            currentGroup = null;
        }
        helpContext = null;
    }

    /**
     * Returns the alert listener.
     *
     * @return the alert listener
     */
    protected AlertListener getAlertListener() {
        return alerts.getListener();
    }

    /**
     * Returns the context.
     *
     * @return the context
     */
    protected Context getContext() {
        return context;
    }

    /**
     * Displays the macros.
     */
    protected void onMacro() {
        MacroDialog dialog = new MacroDialog(context, getHelpContext());
        dialog.show();
    }

    /**
     * Determines if saving has been disabled.
     *
     * @return {@code true} if saves are disabled
     */
    protected boolean isSaveDisabled() {
        return savedDisabled;
    }

    /**
     * Invoked when the editor component changes.
     */
    private void onComponentChange() {
        setComponent(editor.getComponent(), editor.getFocusGroup(), editor.getHelpContext(), false);
    }

    /**
     * Invoked to veto/allow a cancel request.
     *
     * @param action the vetoable action
     */
    private void onCancel(final Vetoable action) {
/*
     TODO - no longer prompt for cancellation due to incorrect isModified() results. See OVPMS-987 for details.

        if (editor != null && editor.isModified() && !savedDisabled) {
            String title = Messages.get("editor.cancel.title");
            String message = Messages.get("editor.cancel.message", editor.getDisplayName());
            final ConfirmationDialog dialog = new ConfirmationDialog(title, message, ConfirmationDialog.YES_NO);
            dialog.addWindowPaneListener(new WindowPaneListener() {
                public void onClose(WindowPaneEvent e) {
                    if (ConfirmationDialog.YES_ID.equals(dialog.getAction())) {
                        action.veto(false);
                    } else {
                        action.veto(true);
                    }
                }
            });
            dialog.show();
        } else {
            action.veto(false);
        }
*/
        action.veto(false);
    }

    /**
     * An {@link IMObjectEditorOperation} that supports reloading the editor if the operation fails.
     */
    protected abstract class ReloadingEditorOperation<T extends IMObjectEditor>
            extends AbstractIMObjectEditorOperation<T> {

        /**
         * Constructs an {@link ReloadingEditorOperation}.
         *
         * @param formatter the failure formatter
         */
        public ReloadingEditorOperation(FailureFormatter formatter) {
            super(formatter);
        }

        /**
         * Invoked to reload the object if the operation fails.
         * <p/>
         * This implementation delegates to {@link AbstractEditDialog#reload(IMObjectEditor)}.
         *
         * @param editor the editor
         * @return {@code true} if the object was reloaded, otherwise {@code false}
         */
        @Override
        protected boolean reload(T editor) {
            return AbstractEditDialog.this.reload(editor);
        }

        /**
         * Invoked after an error has occurred, and the editor has been successfully reloaded.
         * <p/>
         * This implementation delegates to {@link AbstractEditDialog#reloaded(String, String, IMObjectEditor)}.
         *
         * @param title     the message title
         * @param message   the message
         * @param oldEditor the previous instance of the editor
         * @param listener  the listener to notify when error handling is complete. May be {@code null}
         */
        @Override
        protected void reloaded(String title, String message, T oldEditor, Runnable listener) {
            AbstractEditDialog.this.reloaded(title, message, oldEditor);
        }

        /**
         * Invoked when the operation fails and the editor cannot be reverted.
         * <p/>
         * This implementation displays the error and disables buttons.
         *
         * @param editor    the editor
         * @param exception the cause of the failure
         * @param listener  the listener to notify when error handling is complete. May be {@code null}
         */
        @Override
        protected void unrecoverableFailure(T editor, Throwable exception, Runnable listener) {
            super.unrecoverableFailure(editor, exception, listener);
            saveFailed();
        }
    }

    /**
     * An {@link IMObjectEditorOperation} that supports reloading the editor if the save fails.
     */
    private class ReloadingSaveOperation extends ReloadingEditorOperation<IMObjectEditor> {

        public ReloadingSaveOperation() {
            super(new SaveFailureFormatter());
        }

        @Override
        public void apply(IMObjectEditor editor) {
            AbstractEditDialog.this.doSave(editor);
        }
    }
}