/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.contact;

import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.party.Contact;
import org.openvpms.web.component.im.edit.AbstractCollectionPropertyEditor;
import org.openvpms.web.component.im.edit.IMObjectEditor;
import org.openvpms.web.component.im.edit.IMObjectTableCollectionEditor;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.util.IMObjectHelper;
import org.openvpms.web.component.property.CollectionProperty;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.Validator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Edits a collection of contacts.
 * <p/>
 * This:
 * <ul>
 * <li>ensures that only one contact of a particular type can be 'preferred'.</li>
 * <li>removes any new contacts that have been added, but are not modified. This behaviour is enabled via
 * {@link #setExcludeUnmodifiedContacts}</li>
 * </ul>
 *
 * @author Tim Anderson
 */
public class ContactCollectionEditor extends IMObjectTableCollectionEditor {

    /**
     * Preferred node identifier.
     */
    private static final String PREFERRED = "preferred";

    /**
     * Constructs a {@link ContactCollectionEditor}.
     *
     * @param property the collection property
     * @param object   the object being edited
     * @param context  the layout context
     */
    public ContactCollectionEditor(CollectionProperty property, IMObject object, LayoutContext context) {
        super(new ContactCollectionPropertyEditor(property), object, context);
    }

    /**
     * Determines if new contacts that have not been modified should be excluded from save.
     *
     * @param exclude if {@code true}, don't save contacts that are new, and have not been modified
     */
    public void setExcludeUnmodifiedContacts(boolean exclude) {
        getCollectionPropertyEditor().setExcludeUnmodified(exclude);
    }

    /**
     * Creates a new object, subject to collection cardinality constraints. This must be registered with the collection.
     * <p/>
     * This implementation will return any existing unmodified contact of the requested archetype.
     *
     * @param archetype the archetype short name. May be {@code null}
     * @return a new object, or {@code null} if the object can't be created
     */
    @Override
    public IMObject create(String archetype) {
        Contact contact = (Contact) getCollectionPropertyEditor().getUnmodified(archetype);
        if (contact == null || getEditor(contact).isModified()) {
            contact = (Contact) super.create(archetype);
            if (contact != null) {
                boolean preferred = isPreferred(contact);
                if (preferred) {
                    for (IMObject object : getCurrentObjects()) {
                        if (object.getArchetype().equals(contact.getArchetype()) && isPreferred(object)) {
                            IMObjectBean bean = getBean(contact);
                            bean.setValue(PREFERRED, false);
                            break;
                        }
                    }
                }
            }
        }
        return contact;
    }

    /**
     * Invoked when the current editor is modified.
     */
    @Override
    protected void onCurrentEditorModified() {
        IMObjectEditor editor = getCurrentEditor();
        if (editor != null) {
            Contact current = (Contact) editor.getObject();
            Property property = editor.getProperty(PREFERRED);
            if (property != null && property.getBoolean()) {
                for (IMObject c : getCurrentObjects()) {
                    Contact contact = (Contact) c;
                    if (!current.equals(contact) && current.getArchetype().equals(contact.getArchetype())) {
                        IMObjectEditor contactEditor = getEditor(contact);
                        if (contactEditor != null) {
                            contactEditor.getProperty(PREFERRED).setValue(false);
                        }
                    }
                }
            }
        }
        super.onCurrentEditorModified();
    }

    /**
     * Returns the collection property editor.
     *
     * @return the collection property editor
     */
    @Override
    protected ContactCollectionPropertyEditor getCollectionPropertyEditor() {
        return (ContactCollectionPropertyEditor) super.getCollectionPropertyEditor();
    }

    /**
     * Edit an object.
     *
     * @param object the object to edit
     * @return the editor
     */
    @Override
    protected IMObjectEditor edit(IMObject object) {
        IMObjectEditor editor = super.edit(object);
        if (excludedFromValidation(editor) && !editor.isValid()) {
            enableNavigation(true);
        }
        return editor;
    }

    /**
     * Adds any object being edited to the collection, if it is valid.
     * <p/>
     * This implementation overrides the default behaviour by ignoring objects that are new and unmodified if they are
     * excluded from validation and saving.
     *
     * @param validator the validator
     * @return {@code true} if the object is valid, otherwise {@code false}
     */
    @Override
    protected boolean addCurrentEdits(Validator validator) {
        boolean valid = true;
        IMObjectEditor editor = getCurrentEditor();
        if (editor != null) {
            if (!excludedFromValidation(editor)) {
                valid = editor.validate(validator);
                if (valid) {
                    addEdited(editor);
                }
            } else {
                // add it even if it is invalid. It will be removed on save, if it doesn't change
                addEdited(editor);
            }
        }
        return valid;
    }

    /**
     * Determines if an editor is excluded from validation.
     *
     * @param editor the editor
     * @return {@code true} if the editor is excluded from validation
     */
    private boolean excludedFromValidation(IMObjectEditor editor) {
        return getCollectionPropertyEditor().excludeUnmodified() && editor.getObject().isNew()
               && !editor.isModified();
    }

    /**
     * Determines if a contact is preferred.
     *
     * @param contact the contact
     * @return {@code true} if the contact is preferred
     */
    private boolean isPreferred(IMObject contact) {
        IMObjectBean bean = getBean(contact);
        return bean.hasNode(PREFERRED) && bean.getBoolean(PREFERRED);
    }

    private static class ContactCollectionPropertyEditor extends AbstractCollectionPropertyEditor {

        /**
         * New objects that may not have been modified.
         */
        private final Set<IMObject> pending = new HashSet<>();

        private final Map<IMObject, IMObjectEditor> pendingEditors = new HashMap<>();

        /**
         * Determines if new, unmodified objects should be excluded.
         */
        private boolean excludeUnmodified;

        /**
         * Constructs a {@link ContactCollectionPropertyEditor}.
         *
         * @param property the collection property
         */
        ContactCollectionPropertyEditor(CollectionProperty property) {
            super(property);
        }

        /**
         * Returns the objects in the collection.
         *
         * @return the objects in the collection
         */
        @Override
        @SuppressWarnings("unchecked")
        public List<IMObject> getObjects() {
            List<IMObject> result;
            if (pending.isEmpty()) {
                result = super.getObjects();
            } else {
                result = new ArrayList<>(super.getObjects());
                for (IMObject object : pending) {
                    if (!result.contains(object)) {
                        result.add(object);
                    }
                }
            }
            return result;
        }

        /**
         * Adds an object to the collection, if it doesn't exist.
         *
         * @param object the object to add
         * @return {@code true} if the object was added, otherwise {@code false}
         */
        @Override
        public boolean add(IMObject object) {
            boolean added;
            if (excludeUnmodified && object.isNew()) {
                added = pending.add(object);
                if (added) {
                    resetValid();
                }
            } else {
                added = super.add(object);
            }
            return added;
        }

        /**
         * Removes an object from the collection.
         * This removes any associated editor.
         *
         * @param object the object to remove
         * @return {@code true} if the object was removed
         */
        @Override
        public boolean remove(IMObject object) {
            boolean removed = super.remove(object);
            if (pending.remove(object)) {
                removed = true;
                resetValid();
            }
            return removed;
        }

        /**
         * Determines if the collection has been modified.
         *
         * @return {@code true} if the collection has been modified
         */
        @Override
        public boolean isModified() {
            boolean modified = super.isModified();
            if (!modified && !pendingEditors.isEmpty()) {
                for (IMObjectEditor editor : pendingEditors.values()) {
                    if (editor.isModified()) {
                        modified = true;
                        break;
                    }
                }
            }
            return modified;
        }

        /**
         * Clears the modified status of the object.
         */
        @Override
        public void clearModified() {
            super.clearModified();
            for (IMObjectEditor editor : pendingEditors.values()) {
                editor.clearModified();
            }
        }

        /**
         * Returns the editor associated with an object in the collection.
         *
         * @param object the object
         * @return the associated editor, or {@code null} if none is found
         */
        @Override
        public IMObjectEditor getEditor(IMObject object) {
            IMObjectEditor editor = pendingEditors.get(object);
            if (editor == null) {
                editor = super.getEditor(object);
            }
            return editor;
        }

        /**
         * Returns the editors.
         * <p>
         * There may be fewer editors than there are objects in the collection,
         * as objects may not have an associated editor.
         *
         * @return the editors
         */
        @Override
        public Collection<IMObjectEditor> getEditors() {
            Collection<IMObjectEditor> result;
            if (pendingEditors.isEmpty()) {
                result = super.getEditors();
            } else {
                result = new ArrayList<>(super.getEditors());
                for (IMObjectEditor object : pendingEditors.values()) {
                    if (!result.contains(object)) {
                        result.add(object);
                    }
                }
            }
            return result;
        }

        /**
         * Associates an object in the collection with an editor. The editor
         * will be responsible for saving/removing it.
         *
         * @param object the object
         * @param editor the editor. Use {@code null} to remove an association
         */
        @Override
        public void setEditor(IMObject object, IMObjectEditor editor) {
            if (object.isNew()) {
                if (editor == null) {
                    pendingEditors.remove(object);
                } else {
                    pendingEditors.put(object, editor);
                }
            } else {
                super.setEditor(object, editor);
            }
        }

        /**
         * Validates the object.
         *
         * @param validator the validator
         * @return {@code true} if the object and its descendants are valid otherwise {@code false}
         */
        @Override
        protected boolean doValidation(Validator validator) {
            addPending();
            boolean result = validator.validate(getProperty());
            if (result) {
                // validate the objects, excluding any that are still pending
                result = validate(validator, super.getObjects());
            }
            return result;
        }

        /**
         * Removes an object from the set of objects to save.
         * This removes any associated editor.
         *
         * @param object the object to remove
         * @return the associated editor, or {@code null} if there is none
         */
        @Override
        protected IMObjectEditor removeEdited(IMObject object) {
            IMObjectEditor editor = super.removeEdited(object);
            if (editor == null) {
                editor = pendingEditors.remove(object);
                if (editor != null) {
                    resetValid(false);
                }
            }
            return editor;
        }

        /**
         * Saves the collection.
         *
         * @throws OpenVPMSException if the save fails
         */
        @Override
        protected void doSave() {
            addPending();
            super.doSave();
        }

        /**
         * Add any pending objects that have been modified, so they will be validated and saved.
         */
        private void addPending() {
            for (IMObject object : pending.toArray(new IMObject[0])) {
                IMObjectEditor editor = pendingEditors.get(object);
                if (editor != null && editor.isModified()) {
                    // add the contact to the collection, and associate it with the editor
                    getProperty().add(object);
                    super.setEditor(object, editor);
                    pendingEditors.remove(object);
                    pending.remove(object);
                }
            }
        }

        /**
         * Determines if new, unmodified objects should be excluded from validation and saving.
         *
         * @param excludeUnmodified if {@code true}, exclude unmodified objects from validation and saving
         */
        void setExcludeUnmodified(boolean excludeUnmodified) {
            this.excludeUnmodified = excludeUnmodified;
        }

        /**
         * Determines if new, unmodified objects should be excluded from validation and saving.
         *
         * @return {@code true} if unmodified objects are excluded from validation and saving
         */
        boolean excludeUnmodified() {
            return excludeUnmodified;
        }

        /**
         * Returns the first unmodified object with the specified short name.
         *
         * @param archetype the contact archetype
         * @return the first unmodified object, or {@code null} if none is found
         */
        IMObject getUnmodified(String archetype) {
            return archetype != null ? IMObjectHelper.getObject(archetype, pending) : null;
        }
    }
}
