/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.bound;

import org.openvpms.web.component.property.Modifiable;
import org.openvpms.web.component.property.ModifiableListener;
import org.openvpms.web.component.property.Property;

import java.util.Objects;


/**
 * Helper to bind a property to a field.
 *
 * @author Tim Anderson
 */
public abstract class Binder {

    /**
     * The property.
     */
    private final Property property;

    /**
     * Listener for property updates.
     */
    private final ModifiableListener listener;

    /**
     * Determines if {@code listener} has been registered with the property.
     */
    private boolean hasListener = false;

    /**
     * Determines if the bind() method has been invoked to bind the field to the property.
     */
    private boolean bound = false;


    /**
     * Constructs a {@link Binder}.
     * <p/>
     * This binds to the property.
     *
     * @param property the property to bind
     */
    public Binder(Property property) {
        this(property, true);
    }

    /**
     * Constructs a {@link Binder}.
     *
     * @param property the property to bind
     * @param bind     if {@code true} bind the property
     */
    public Binder(Property property, boolean bind) {
        this.property = property;
        listener = new ModifiableListener() {
            public void modified(Modifiable modifiable) {
                setField();
            }
        };
        if (bind) {
            bind();
        }
    }

    /**
     * Updates the property from the field.
     */
    public void setProperty() {
        boolean listener = hasListener;
        if (listener) {
            // remove the listener to avoid cyclic notifications
            removeModifiableListener();
        }
        try {
            setProperty(property);
        } finally {
            if (listener) {
                addModifiableListener();
            }
        }
    }

    /**
     * Updates the field from the property.
     */
    public void setField() {
        setFieldValue(property.getValue());
    }

    /**
     * Registers the binder with the property to receive updates.
     */
    public void bind() {
        if (!bound) {
            setField(); // update the field from the property
            addModifiableListener();
            bound = true;
        }
    }

    /**
     * Deregisters the binder from the property.
     */
    public void unbind() {
        if (bound) {
            removeModifiableListener();
            bound = false;
        }
    }

    /**
     * Returns the property.
     *
     * @return the property
     */
    public Property getProperty() {
        return property;
    }

    /**
     * Updates the property from the field.
     *
     * @param property the property to update
     * @return {@code true} if the property was updated
     */
    protected boolean setProperty(Property property) {
        Object fieldValue = getFieldValue();
        boolean result = property.setValue(fieldValue);
        if (result) {
            Object propertyValue = property.getValue();
            if (!Objects.equals(fieldValue, propertyValue)) {
                setField();
            }
        }
        return result;
    }

    /**
     * Returns the value of the field.
     *
     * @return the value of the field
     */
    protected abstract Object getFieldValue();

    /**
     * Sets the value of the field.
     *
     * @param value the value to set
     */
    protected abstract void setFieldValue(Object value);

    /**
     * Determines if the binder is bound to the property.
     *
     * @return {@code true} if the binder is bound, otherwise {@code false}
     */
    protected boolean isBound() {
        return bound;
    }

    /**
     * Registers the listener with the property, to receive notification when the property changes.
     */
    private void addModifiableListener() {
        property.addModifiableListener(listener, 0);
        hasListener = true;
    }

    /**
     * Deregisters the listener from the property.
     */
    private void removeModifiableListener() {
        property.removeModifiableListener(listener);
        hasListener = false;
    }
}
