/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.app;

import org.openvpms.component.model.object.IMObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Application context information.
 *
 * @author Tim Anderson
 */
public class GlobalContext extends AbstractContext implements ContextHistory {

    /**
     * The context listeners.
     */
    private final List<ContextListener> contextListeners = new ArrayList<>();

    /**
     * Listeners for particular archetypes.
     */
    private final Map<String, List<ContextListener>> archetypeListeners = new HashMap<>();

    /**
     * The context history.
     */
    private final Map<String, SelectionHistory> history = new HashMap<>();


    /**
     * Default constructor.
     */
    public GlobalContext() {
        super();
    }

    /**
     * Adds a listener to be notified of changes to an archetype.
     *
     * @param archetype the archetype
     * @param listener  the listener to add
     */
    public void addListener(String archetype, ContextListener listener) {
        List<ContextListener> listeners = archetypeListeners.computeIfAbsent(archetype, k -> new ArrayList<>());
        listeners.add(listener);
    }

    /**
     * Removes a listener.
     *
     * @param archetype the archetype
     * @param listener  the listener to remove
     */
    public void removeListener(String archetype, ContextListener listener) {
        List<ContextListener> listeners = archetypeListeners.get(archetype);
        if (listeners != null) {
            listeners.remove(listener);
        }
    }

    /**
     * Adds a listener.
     *
     * @param listener the listener to add
     */
    public void addListener(ContextListener listener) {
        contextListeners.add(listener);
    }

    /**
     * Removes a listener.
     *
     * @param listener the listener to remove
     */
    public void removeListener(ContextListener listener) {
        contextListeners.remove(listener);
    }

    /**
     * Sets a context object.
     *
     * @param key    the context key
     * @param object the object
     */
    @Override
    public void setObject(String key, IMObject object) {
        IMObject current = getObject(key);
        if (current != object) {
            // only update the context if the objects have different instances,
            // to avoid cyclic notifications
            super.setObject(key, object);
            if (object != null) {
                updateHistory(key, object);
            }
            notifyListeners(key, object);
        }
    }

    /**
     * Returns the history for a short name.
     *
     * @param shortName the archetype short name
     * @return the selection history
     */
    public SelectionHistory getHistory(String shortName) {
        SelectionHistory result = history.get(shortName);
        return result != null ? result : new SelectionHistory(this);
    }

    /**
     * Registers selection history for a short name.
     *
     * @param shortName the short name
     * @param history   the history
     */
    public void setHistory(String shortName, SelectionHistory history) {
        this.history.put(shortName, history);
    }

    /**
     * Updates the history.
     *
     * @param key    the context key
     * @param object the context value
     */
    private void updateHistory(String key, IMObject object) {
        SelectionHistory selectionHistory = history.get(key);
        if (selectionHistory == null) {
            selectionHistory = new SelectionHistory(this);
            history.put(key, selectionHistory);
        }
        selectionHistory.add(object);
    }

    /**
     * Notifies listeners of a change of objects.
     *
     * @param key   the context key
     * @param value the context value
     */
    private void notifyListeners(String key, IMObject value) {
        List<ContextListener> listeners = archetypeListeners.get(key);
        if (listeners != null) {
            notifyListeners(listeners, key, value);
        }
        notifyListeners(contextListeners, key, value);
    }

    /**
     * Notifies listeners of a change of objects.
     *
     * @param key   the context key
     * @param value the context value
     */
    private void notifyListeners(List<ContextListener> listeners, String key, IMObject value) {
        ContextListener[] list = listeners.toArray(new ContextListener[0]);
        for (ContextListener listener : list) {
            listener.changed(key, value);
        }
    }
}
