/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.app;

import org.openvpms.archetype.rules.party.ContactArchetypes;
import org.openvpms.archetype.rules.party.Contacts;
import org.openvpms.archetype.rules.party.PurposeMatcher;
import org.openvpms.component.model.party.Contact;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.component.system.common.util.Variables;
import org.openvpms.web.component.im.contact.ContactHelper;
import org.openvpms.web.component.macro.MacroVariables;
import org.openvpms.web.component.mail.MailContext;
import org.openvpms.web.system.ServiceHelper;

import java.util.List;

/**
 * An {@link MailContext} that sources email addresses from objects held by the supplied {@link Context}.
 *
 * @author Tim Anderson
 */
public abstract class ContextMailContext extends AbstractMailContext {

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The context to use.
     */
    private final Context context;

    /**
     * The purpose used to select the default from-address. If {@code null}, then the preferred from-address will
     * be used.
     */
    private final String fromPurpose;

    /**
     * The variables.
     */
    private Variables variables;

    /**
     * Constructs a {@link ContextMailContext}.
     *
     * @param context the context
     */
    public ContextMailContext(Context context) {
        this(context, null);
    }

    /**
     * Constructs a {@link ContextMailContext}.
     *
     * @param context the context
     * @param fromPurpose if non-null, use the email address with the specified purpose as the from-address, if any
     */
    public ContextMailContext(Context context, String fromPurpose) {
        this.context = context;
        this.fromPurpose = fromPurpose;
        this.service = ServiceHelper.getArchetypeService();
    }


    /**
     * Returns the available 'from' email addresses.
     * <p/>
     * This implementation returns the email contacts from the current practice location if any, or the practice.
     *
     * @return the 'from' email addresses
     */
    public List<Contact> getFromAddresses() {
        List<Contact> result = ContactHelper.getEmailContacts(context.getLocation(), service);
        if (result.isEmpty()) {
            result = ContactHelper.getEmailContacts(context.getPractice(), service);
        }
        if (fromPurpose != null && !result.isEmpty()) {
            PurposeMatcher matcher = new PurposeMatcher(ContactArchetypes.EMAIL, fromPurpose, service);
            Contact match = Contacts.find(result, matcher);
            if (match != null) {
                int index = result.indexOf(match);
                if (index > 0) {
                    result.remove(index);
                    result.add(0, match);
                }
            }
        }
        return result;
    }

    /**
     * Returns variables to be used in macro expansion.
     *
     * @return an instance of {@link MacroVariables}
     */
    @Override
    public Variables getVariables() {
        if (variables == null) {
            variables = new MacroVariables(new ReloadingContext(context), service, ServiceHelper.getLookupService());
        }
        return variables;
    }

    /**
     * Returns the context.
     *
     * @return the context
     */
    protected Context getContext() {
        return context;
    }

    /**
     * Returns the archetype service.
     *
     * @return the archetype service
     */
    protected ArchetypeService getService() {
        return service;
    }
}
