/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.smartflow.client;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.archetype.rules.doc.DocumentHandlers;
import org.openvpms.archetype.rules.patient.MedicalRecordRules;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.security.crypto.PasswordEncryptor;
import org.openvpms.component.service.lookup.LookupService;
import org.openvpms.smartflow.i18n.FlowSheetMessages;

import java.util.TimeZone;

/**
 * Factory for Smart Flow Sheet services.
 *
 * @author Tim Anderson
 */
public class FlowSheetServiceFactory {

    /**
     * The Smart Flow Sheet URL.
     */
    private final String url;

    /**
     * The emrApiKey submitted with each request.
     */
    private final String emrApiKey;

    /**
     * The archetype service.
     */
    private final IArchetypeService service;

    /**
     * The lookup service.
     */
    private final LookupService lookups;

    /**
     * The document handlers.
     */
    private final DocumentHandlers handlers;

    /**
     * The medical record rules.
     */
    private final MedicalRecordRules rules;

    /**
     * The encryptor for decrypting API keys.
     */
    private final PasswordEncryptor encryptor;

    /**
     * Constructs an {@link FlowSheetServiceFactory}.
     *
     * @param url       the Smart Flow Sheet URL
     * @param emrApiKey the emrApiKey submitted with each request
     * @param service   the archetype service
     * @param lookups   the lookup service
     * @param handlers  the document handlers
     * @param rules     the medical record rules
     * @param encryptor the encryptor for decrypting API keys
     */
    public FlowSheetServiceFactory(String url, String emrApiKey, IArchetypeService service, LookupService lookups,
                                   DocumentHandlers handlers, MedicalRecordRules rules, PasswordEncryptor encryptor) {
        this.url = url;
        this.emrApiKey = emrApiKey;
        this.service = service;
        this.lookups = lookups;
        this.handlers = handlers;
        this.rules = rules;
        this.encryptor = encryptor;
    }

    /**
     * Determines if a practice location has a clinic API key.
     *
     * @param location the practice location
     * @return {@code true} if the location has a key
     */
    public boolean isSmartFlowSheetEnabled(Party location) {
        return getClinicAPIKey(location) != null;
    }

    /**
     * Returns the clinic API key for the practice.
     *
     * @param location the practice location
     * @return the clinic API key, or {@code null} if none exists
     * @throws FlowSheetException if the key exists but cannot be decrypted
     */
    public String getClinicAPIKey(Party location) {
        String result = null;
        if (location != null) {
            IMObjectBean bean = service.getBean(location);
            result = StringUtils.trimToNull(bean.getString("smartFlowSheetKey"));
            if (result != null) {
                try {
                    result = encryptor.decrypt(result);
                } catch (Throwable exception) {
                    throw new FlowSheetException(FlowSheetMessages.failedToDecryptAPIKey(location), exception);
                }
            }
        }
        return result;
    }

    /**
     * Creates a {@link HospitalizationService} for the specified practice location.
     *
     * @param location the practice location
     * @return a new {@link HospitalizationService}
     */
    public HospitalizationService getHospitalizationService(Party location) {
        String apiKey = getRequiredClinicAPIKey(location);
        return getHospitalizationService(apiKey);
    }

    /**
     * Creates a {@link HospitalizationService} for the specified practice location.
     *
     * @param apiKey the clinic API key
     * @return a new {@link HospitalizationService}
     */
    public HospitalizationService getHospitalizationService(String apiKey) {
        return new HospitalizationService(url, emrApiKey, apiKey, TimeZone.getDefault(), service, lookups, handlers,
                                          rules);
    }

    /**
     * Creates a {@link ReferenceDataService} for the specified practice location.
     *
     * @param location the practice location
     * @return a new {@link ReferenceDataService}
     */
    public ReferenceDataService getReferenceDataService(Party location) {
        String clinicKey = getRequiredClinicAPIKey(location);
        return new ReferenceDataService(url, emrApiKey, clinicKey, TimeZone.getDefault(), location, service);
    }

    /**
     * Creates a {@link InventoryService} for the specified practice location.
     *
     * @param location the practice location
     * @return a new {@link InventoryService}
     */
    public InventoryService getInventoryService(Party location) {
        String clinicKey = getRequiredClinicAPIKey(location);
        return new InventoryService(url, emrApiKey, clinicKey, TimeZone.getDefault(), service, lookups);
    }

    /**
     * Returns the clinic API key for the practice, throwing an exception if it doesn't exist
     *
     * @param location the practice location
     * @return the clinic API key
     * @throws FlowSheetException if the API key does not exist or cannot be decrypted
     */
    protected String getRequiredClinicAPIKey(Party location) {
        String clinicKey = getClinicAPIKey(location);
        if (clinicKey == null) {
            throw new FlowSheetException(FlowSheetMessages.notConfigured(location));
        }
        return clinicKey;
    }

}
