/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2022 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.report.jasper;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.design.JasperDesign;
import net.sf.jasperreports.engine.xml.JRXmlLoader;
import net.sf.jasperreports.engine.xml.JRXmlWriter;
import org.apache.commons.io.output.CountingOutputStream;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.openvpms.archetype.rules.doc.AbstractDocumentHandler;
import org.openvpms.archetype.rules.doc.DocumentArchetypes;
import org.openvpms.archetype.rules.doc.DocumentException;
import org.openvpms.component.business.domain.im.document.Document;
import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.report.i18n.ReportMessages;
import org.xml.sax.SAXParseException;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.zip.DeflaterOutputStream;


/**
 * Jasper Reports document handler.
 *
 * @author Tim Anderson
 */
public class JRXMLDocumentHandler extends AbstractDocumentHandler {

    /**
     * Constructs a {@code JRXMLDocumentHandler}.
     *
     * @param service the archetype service
     */
    public JRXMLDocumentHandler(ArchetypeService service) {
        super(DocumentArchetypes.DEFAULT_DOCUMENT, service);
    }

    /**
     * Determines if this handler supports a document.
     *
     * @param name     the document name
     * @param mimeType the mime type of the document. May be {@code null}
     * @return {@code true} if this handler supports the document
     */
    @Override
    public boolean canHandle(String name, String mimeType) {
        return name.endsWith(".jrxml");
    }

    /**
     * Determines if this handler supports a document.
     *
     * @param name      the document name
     * @param archetype the document archetype short name
     * @param mimeType  the mime type of the document. May be {@code null}
     * @return {@code true} if this handler supports the document
     */
    @Override
    public boolean canHandle(String name, String archetype, String mimeType) {
        boolean result = super.canHandle(name, archetype, mimeType);
        return result && name.endsWith(".jrxml");
    }

    /**
     * Creates a new {@link Document} from a stream.
     * <p/>
     * This verifies that the file is a valid .jrxml, and regenerates it using the current JasperReport version.
     * <p/>
     * As a result, the CRC and size may change.
     *
     * @param name     the document name
     * @param stream   a stream representing the document content
     * @param mimeType the mime type of the document. May be {@code null}
     * @param size     the size of stream
     * @return a new document
     * @throws DocumentException         if the document can't be created
     * @throws ArchetypeServiceException for any archetype service error
     * @throws JRXMLDocumentException    if the document version cannot be parsed
     */
    public Document create(String name, InputStream stream, String mimeType, int size) {
        Document document;
        JasperDesign design;
        try {
            design = JRXmlLoader.load(stream);
        } catch (JRException exception) {
            if (ExceptionUtils.getRootCause(exception) instanceof SAXParseException) {
                if (name == null) {
                    name = "file";
                }
                throw new JRXMLDocumentException(ReportMessages.failedToReadJasperReport(name), exception);
            } else {
                throw new DocumentException(DocumentException.ErrorCode.ReadError, exception, name);
            }
        }
        try {
            ByteArrayOutputStream bytes = new ByteArrayOutputStream();
            DeflaterOutputStream deflater = new DeflaterOutputStream(bytes);
            CountingOutputStream output = new CountingOutputStream(deflater);
            JRXmlWriter.writeReport(design, output, "UTF-8");
            deflater.close();
            document = create(name, bytes.toByteArray(), "text/xml", output.getCount());
        } catch (Exception exception) {
            throw new DocumentException(DocumentException.ErrorCode.WriteError, exception, name);
        }
        return document;
    }

}
