/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.report.jasper;

import net.sf.jasperreports.engine.JRRewindableDataSource;
import org.apache.commons.jxpath.Functions;
import org.openvpms.archetype.rules.doc.DocumentHandlers;
import org.openvpms.archetype.rules.settings.Settings;
import org.openvpms.component.business.service.archetype.helper.ResolvingPropertySet;
import org.openvpms.component.model.document.Document;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.component.service.lookup.LookupService;
import org.openvpms.component.system.common.util.PropertySet;
import org.openvpms.report.Parameters;
import org.openvpms.report.ReportException;

import java.util.Map;


/**
 * A {@link JasperIMReport} that uses pre-defined templates, operating on collections of {@link IMObject}s.
 *
 * @author Tim Anderson
 */
public class TemplatedJasperIMObjectReport extends AbstractTemplatedJasperIMReport<IMObject> {

    /**
     * Constructs a {@link TemplatedJasperIMObjectReport}.
     *
     * @param name      the document template name, used in error reporting
     * @param document  the document containing the JasperReport
     * @param service   the archetype service
     * @param lookups   the lookup service
     * @param handlers  the document handlers
     * @param functions the JXPath extension functions
     * @param settings  the settings
     * @throws ReportException if the report cannot be created
     */
    public TemplatedJasperIMObjectReport(String name, Document document, ArchetypeService service, LookupService lookups,
                                         DocumentHandlers handlers, Functions functions, Settings settings) {
        super(name, document, service, lookups, handlers, functions, settings);
    }

    /**
     * Creates a data source for a collection of objects.
     *
     * @param objects    an iterator over the collection of objects
     * @param parameters a map of parameter names and their values, to pass to the report. May be {@code null}
     * @param fields     a map of additional field names and their values, to pass to the report. May be {@code null}
     * @return a new data source
     */
    @Override
    protected JRRewindableDataSource createDataSource(Iterable<IMObject> objects, Map<String, Object> parameters,
                                                      Map<String, Object> fields) {
        ArchetypeService service = getArchetypeService();
        LookupService lookups = getLookupService();
        PropertySet f = (fields != null) ? new ResolvingPropertySet(fields, service, lookups) : null;
        Parameters p = (parameters) != null ? new Parameters(parameters) : null;
        return new IMObjectCollectionDataSource(objects, p, f, getName(), service, lookups, getDocumentHandlers(),
                                                getFunctions());
    }

}
