/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.plugin.internal.service.security;

import org.openvpms.archetype.rules.practice.PracticeService;
import org.openvpms.component.business.service.security.AuthenticationContext;
import org.openvpms.component.business.service.security.RunAs;
import org.openvpms.component.model.user.User;
import org.openvpms.component.service.archetype.ArchetypeService;

/**
 * Runs an operation as the current user if present, or the {@link PracticeService#getServiceUser()
 * practice service user} if not.
 * <p/>
 * This is designed to be used for code invoked by plugins where the security context may or may not be set,
 * but a user is required in order to invoke {@link ArchetypeService} save operations, directly, or indirectly.
 * <p/>
 * The problem this is working around is that business rules all use {@link ArchetypeService} implementations which
 * rely on the Spring security context being populated, but plugins often operate from threads where the context isn't
 * set. There is no facility to assign users to plugins, or allow plugins to set the current user.
 * <p/>
 * An alternative solution would be to assign users to bundles, and provide a service that can:
 * <ul>
 *     <li>run an operation as the current user (the default); or</li>
 *     <li>run an operation as the bundle's user (where no user is present, such as for background threads); or</li>
 *     <li>run an operation as the another user, subject to permission checks</li>
 * </ul>
 * This should be used in conjunction with a thread factory for plugins, so the user is set for any threads that plugins
 * may create. TODO
 *
 * @author Tim Anderson
 */
public class RunAsService {

    /**
     * The practice service.
     */
    private final PracticeService practiceService;

    /**
     * The authentication context.
     */
    private final AuthenticationContext context;

    /**
     * Constructs a {@link RunAsService}.
     *
     * @param context         the authentication context
     * @param practiceService the practice service
     */
    public RunAsService(AuthenticationContext context, PracticeService practiceService) {
        this.context = context;
        this.practiceService = practiceService;
    }

    /**
     * Runs an operation as the current user if present, or the {@link PracticeService#getServiceUser()
     * practice service user} if not.
     *
     * @param runnable the operation to run
     * @throws IllegalStateException if there is no user available
     */
    public void run(Runnable runnable) {
        User user = context.getUser();
        if (user != null) {
            runnable.run();
        } else {
            RunAs.run(getUser(), runnable);
        }
    }

    /**
     * Returns the fallback practice service user.
     *
     * @return the user
     * @throws IllegalStateException if none is configured
     */
    private User getUser() {
        User user;
        user = practiceService.getServiceUser();
        if (user == null) {
            throw new IllegalStateException("The Service User has not been configured on the Practice");
        }
        return user;
    }
}