/*
 * Decompiled with CFR 0.152.
 */
package org.openvpms.tools.data.loader;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import javax.xml.stream.Location;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import org.apache.commons.lang3.StringUtils;
import org.openvpms.component.business.domain.im.archetype.descriptor.ArchetypeDescriptor;
import org.openvpms.component.business.domain.im.common.IMObject;
import org.openvpms.component.business.domain.im.common.IMObjectReference;
import org.openvpms.component.business.domain.im.party.Contact;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.object.Identity;
import org.openvpms.component.model.object.Relationship;
import org.openvpms.tools.data.loader.ArchetypeDataLoaderException;
import org.openvpms.tools.data.loader.Data;
import org.openvpms.tools.data.loader.DeferredUpdater;
import org.openvpms.tools.data.loader.LoadCache;
import org.openvpms.tools.data.loader.LoadContext;
import org.openvpms.tools.data.loader.LoadState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DataLoader {
    private final LoadCache cache;
    private final IArchetypeService service;
    private final boolean verbose;
    private final boolean validateOnly;
    private final Map<String, Long> statistics;
    private final Logger log = LoggerFactory.getLogger(DataLoader.class);
    private LoadContext context;
    private int batchSize;
    private Map<IMObjectReference, LoadState> queue = new LinkedHashMap<IMObjectReference, LoadState>();
    private List<LoadState> deferred = new ArrayList<LoadState>();

    public DataLoader(int batchSize, IArchetypeService service) {
        this(new LoadCache(), service, false, false, batchSize, new HashMap<String, Long>());
    }

    public DataLoader(LoadCache cache, IArchetypeService service, boolean verbose, boolean validateOnly, int batchSize, Map<String, Long> statistics) {
        this.cache = cache;
        this.service = service;
        this.verbose = verbose;
        this.validateOnly = validateOnly;
        this.batchSize = batchSize;
        this.context = new LoadContext(service, cache, validateOnly);
        this.statistics = statistics;
    }

    public void load(XMLStreamReader reader, String path) throws XMLStreamException {
        Stack<LoadState> stack = new Stack<LoadState>();
        int depth = 0;
        int event = reader.next();
        while (event != 8) {
            switch (event) {
                case 7: {
                    break;
                }
                case 1: {
                    ++depth;
                    String elementName = reader.getLocalName();
                    if ("data".equals(elementName)) {
                        LoadState parent = !stack.isEmpty() ? (LoadState)stack.peek() : null;
                        LoadState state = this.startData(reader, parent, path);
                        if (state == null) break;
                        state.setDepth(depth);
                        stack.push(state);
                        break;
                    }
                    if ("archetype".equals(elementName)) break;
                    throw new ArchetypeDataLoaderException(ArchetypeDataLoaderException.ErrorCode.ErrorInStartElement);
                }
                case 2: {
                    if (!stack.isEmpty() && ((LoadState)stack.peek()).getDepth() == depth) {
                        LoadState current = (LoadState)stack.pop();
                        this.load(current);
                    }
                    --depth;
                    if (!this.verbose) break;
                    this.log.info("[END PROCESSING element=" + reader.getLocalName() + "]");
                    break;
                }
            }
            event = reader.next();
        }
        if (!stack.isEmpty()) {
            throw new IllegalStateException("Stack not empty");
        }
    }

    public void flush() {
        this.save();
        while (this.processDeferred()) {
            this.save();
        }
    }

    public void close() {
        this.flush();
        for (LoadState state : this.queue.values()) {
            Set<IMObjectReference> unsaved = state.getUnsaved();
            if (unsaved.isEmpty()) continue;
            for (IMObjectReference ref : unsaved) {
                if (this.queue.containsKey(ref)) continue;
                String id = this.cache.getId(ref);
                if (id == null) {
                    id = "<unset>";
                }
                this.log.error("Cannot save object, archetype=" + state.getArchetype() + " from path=" + state.getPath() + ", line=" + state.getLineNumber() + ": requires " + unsaved + ", id=" + id);
            }
        }
        for (LoadState state : this.deferred) {
            for (DeferredUpdater deferred : state.getDeferred()) {
                this.log.error("Cannot save object, archetype=" + state.getArchetype() + " from path=" + state.getPath() + ", line=" + state.getLineNumber() + ": requires id=" + deferred.getId());
            }
        }
    }

    public Map<String, Long> getStatistics() {
        return this.statistics;
    }

    public LoadCache getLoadCache() {
        return this.cache;
    }

    private LoadState startData(XMLStreamReader reader, LoadState parent, String path) {
        LoadState current = null;
        Data data = new Data(reader);
        if (this.verbose) {
            String archetype = parent == null ? "none" : parent.getArchetype();
            this.log.info("[START PROCESSING element, parent=" + archetype + "]" + data);
        }
        try {
            current = this.processData(parent, data, path);
        }
        catch (ArchetypeDataLoaderException exception) {
            this.log.error(exception.getMessage());
        }
        catch (Exception exception) {
            Location location = reader.getLocation();
            this.log.error("Error in start element, line " + location.getLineNumber() + ", column " + location.getColumnNumber() + "" + data + "", (Throwable)exception);
        }
        return current;
    }

    private LoadState processData(LoadState parent, Data data, String path) {
        LoadState result = null;
        Location location = data.getLocation();
        String collectionNode = data.getCollection();
        String childId = data.getChildId();
        if (StringUtils.isEmpty((CharSequence)childId)) {
            result = this.create(data, parent, path);
            for (Map.Entry<String, String> attribute : data.getAttributes().entrySet()) {
                String name = attribute.getKey();
                String value = attribute.getValue();
                result.setValue(name, value, this.context);
            }
            if (collectionNode != null) {
                if (parent == null) {
                    throw new ArchetypeDataLoaderException(ArchetypeDataLoaderException.ErrorCode.NoParentForChild, location.getLineNumber(), location.getColumnNumber());
                }
                parent.addChild(collectionNode, result);
            }
        } else {
            if (parent == null) {
                throw new ArchetypeDataLoaderException(ArchetypeDataLoaderException.ErrorCode.NoParentForChild, location.getLineNumber(), location.getColumnNumber());
            }
            if (StringUtils.isEmpty((CharSequence)collectionNode)) {
                throw new ArchetypeDataLoaderException(ArchetypeDataLoaderException.ErrorCode.NoCollectionAttribute, location.getLineNumber(), location.getColumnNumber());
            }
            parent.addChild(collectionNode, childId, this.context);
        }
        return result;
    }

    private LoadState create(Data data, LoadState parent, String path) {
        String shortName = data.getShortName();
        ArchetypeDescriptor descriptor = this.service.getArchetypeDescriptor(shortName);
        Location location = data.getLocation();
        if (descriptor == null) {
            throw new ArchetypeDataLoaderException(ArchetypeDataLoaderException.ErrorCode.InvalidArchetype, location.getLineNumber(), location.getColumnNumber(), shortName);
        }
        IMObject object = this.service.create(descriptor.getType());
        if (object == null) {
            throw new ArchetypeDataLoaderException(ArchetypeDataLoaderException.ErrorCode.InvalidArchetype, location.getLineNumber(), location.getColumnNumber(), shortName);
        }
        if ((object instanceof Relationship || object instanceof Identity || object instanceof Contact) && parent == null) {
            throw new ArchetypeDataLoaderException(ArchetypeDataLoaderException.ErrorCode.ObjectMustBeInCollection, location.getLineNumber(), location.getColumnNumber(), shortName);
        }
        this.cache.add(object, data.getId());
        return new LoadState(parent, object, descriptor, path, data.getLocation().getLineNumber(), data.getId());
    }

    private void load(LoadState state) {
        if (!state.isComplete()) {
            this.deferred.add(state);
        } else {
            String archetype;
            Long count;
            IMObject object = state.getObject();
            if (this.queue(state)) {
                this.processDeferred();
            }
            if ((count = this.statistics.get(archetype = state.getArchetype())) == null) {
                this.statistics.put(archetype, 1L);
            } else {
                this.statistics.put(archetype, count + 1L);
            }
            if (this.verbose) {
                this.log.info("[CREATED]" + object);
            }
        }
    }

    private boolean queue(LoadState state) {
        boolean saved = false;
        IMObject object = state.getObject();
        this.service.deriveValues(object);
        if (this.validateOnly) {
            try {
                this.service.validateObject(object);
            }
            catch (Throwable exception) {
                this.log.error("Failed to validate object, archetype=" + object.getArchetype() + " from path=" + state.getPath() + ", line=" + state.getLineNumber() + ", id=" + state.getId(), exception);
            }
        } else {
            this.queue.put(object.getObjectReference(), state);
            if (this.queue.size() >= this.batchSize) {
                this.save();
                saved = true;
            }
        }
        return saved;
    }

    private void save() {
        Collection<LoadState> values = this.queue.values();
        LoadState[] objects = values.toArray(new LoadState[0]);
        LinkedHashMap<IMObjectReference, IMObject> batch = new LinkedHashMap<IMObjectReference, IMObject>();
        for (LoadState state : objects) {
            LinkedHashMap<IMObjectReference, IMObject> attempt;
            IMObject object = state.getObject();
            IMObjectReference ref = object.getObjectReference();
            if (batch.containsKey(ref) || !this.getPending(ref, attempt = new LinkedHashMap<IMObjectReference, IMObject>(batch))) continue;
            batch = attempt;
        }
        if (!batch.isEmpty()) {
            try {
                this.save(batch);
            }
            catch (OpenVPMSException exception) {
                for (LoadState state : objects) {
                    IMObject object = state.getObject();
                    IMObjectReference ref = object.getObjectReference();
                    if (!this.getPending(ref, batch = new LinkedHashMap())) continue;
                    try {
                        this.save(batch);
                    }
                    catch (OpenVPMSException e) {
                        Set unsaved = batch.keySet();
                        this.queue.keySet().removeAll(unsaved);
                        this.log.error("Failed to save object, archetype=" + object.getArchetypeId().getShortName() + " from path=" + state.getPath() + ", line=" + state.getLineNumber(), (Throwable)e);
                    }
                }
            }
        }
    }

    private boolean processDeferred() {
        boolean processed;
        boolean result = false;
        do {
            LoadState[] states;
            processed = false;
            for (LoadState state : states = this.deferred.toArray(new LoadState[0])) {
                Collection<DeferredUpdater> updaters = state.getDeferred();
                if (!updaters.isEmpty()) {
                    for (DeferredUpdater updater : updaters.toArray(new DeferredUpdater[0])) {
                        String id = updater.getId();
                        IMObjectReference ref = this.context.getReference(id);
                        if (ref == null || !updater.update(ref, this.context)) continue;
                        processed = true;
                        result = true;
                    }
                    continue;
                }
                this.deferred.remove(state);
                this.queue(state);
                result = true;
            }
        } while (processed);
        return result;
    }

    private boolean getPending(IMObjectReference ref, Map<IMObjectReference, IMObject> objects) {
        boolean resolved = true;
        LoadState state = this.queue.get(ref);
        if (state != null) {
            objects.put(ref, state.getObject());
            for (IMObjectReference unsaved : state.getUnsaved()) {
                if (objects.containsKey(unsaved) || this.getPending(unsaved, objects)) continue;
                resolved = false;
                break;
            }
        } else {
            resolved = false;
        }
        return resolved;
    }

    private void save(Map<IMObjectReference, IMObject> objects) {
        Set<IMObjectReference> saved = objects.keySet();
        this.service.save(objects.values());
        this.queue.keySet().removeAll(saved);
        for (IMObject object : objects.values()) {
            IMObjectReference reference = object.getObjectReference();
            this.cache.update(reference);
            this.update(reference, this.queue.values());
            this.update(reference, this.deferred);
        }
    }

    private void update(IMObjectReference reference, Collection<LoadState> states) {
        for (LoadState state : states) {
            if (!state.getUnsaved().contains(reference)) continue;
            state.update(reference);
        }
    }
}

