/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.internal.service;

import org.openvpms.archetype.rules.finance.account.CustomerAccountRules;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.domain.service.object.DomainObjectService;
import org.openvpms.paymentprocessor.processor.PaymentProcessor;
import org.openvpms.paymentprocessor.service.PaymentProcessorService;
import org.openvpms.paymentprocessor.transaction.Transaction;

import static org.openvpms.archetype.rules.act.ActStatus.POSTED;
import static org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorTransactionStatus.SUBMITTED;

/**
 * Checks a payment or refund with payment processor transactions, to determine if they have completed.
 *
 * @author Tim Anderson
 */
public class PaymentProcessorTransactionChecker {

    private final PaymentProcessors paymentProcessors;


    private final CustomerAccountRules rules;

    private final DomainObjectService domainService;

    public PaymentProcessorTransactionChecker(PaymentProcessors paymentProcessors, CustomerAccountRules rules,
                                              DomainObjectService domainService) {
        this.rules = rules;
        this.paymentProcessors = paymentProcessors;
        this.domainService = domainService;
    }

    /**
     * Checks a payment/refund to determine if it has a payment processor transaction that has updated.
     *
     * @param act the payment/refund
     * @return {@code true} if there was a transaction present that updated, otherwise {@code false}
     */
    public boolean check(FinancialAct act) {
        boolean result = false;
        if (!POSTED.equals(act.getStatus())) {
            FinancialAct transactionAct = rules.getPaymentProcessorTransaction(act);
            if (transactionAct != null && SUBMITTED.equals(transactionAct.getStatus())) {
                Transaction transaction = domainService.create(transactionAct, Transaction.class);
                PaymentProcessor paymentProcessor = transaction.getPaymentProcessor();
                PaymentProcessorService service = paymentProcessors.getPaymentProcessor(paymentProcessor);
                PaymentProcessorServiceAdapter adapter = new PaymentProcessorServiceAdapter(service);
                result = adapter.check(transaction);
            }
        }
        return result;
    }
}