/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.internal.service;

import org.openvpms.paymentprocessor.exception.PaymentProcessorException;
import org.openvpms.paymentprocessor.processor.TransactionMode;
import org.openvpms.paymentprocessor.service.PaymentProcessorService;
import org.openvpms.paymentprocessor.service.PaymentRequirements;
import org.openvpms.paymentprocessor.service.RefundRequirements;
import org.openvpms.paymentprocessor.service.ValidationStatus;
import org.openvpms.paymentprocessor.transaction.Payment;
import org.openvpms.paymentprocessor.transaction.Refund;
import org.openvpms.paymentprocessor.transaction.Transaction;

/**
 * Adapter to make the {@link PaymentProcessorService} more convenient to use.
 *
 * @author Tim Anderson
 */
public class PaymentProcessorServiceAdapter {

    /**
     * The payment processor service.
     */
    private final PaymentProcessorService service;

    /**
     * Constructs a {@link PaymentProcessorServiceAdapter}.
     *
     * @param service the payment processor service
     */
    public PaymentProcessorServiceAdapter(PaymentProcessorService service) {
        this.service = service;
    }

    /**
     * Returns the payment requirements for the specified transaction mode.
     *
     * @param mode the transaction mode
     * @return the payment requirements
     */
    public PaymentRequirements getPaymentRequirements(TransactionMode mode) {
        return service.getPaymentCapabilities().getRequirements(mode);
    }

    /**
     * Returns the refund requirements for the specified transaction mode.
     *
     * @param mode the transaction mode
     * @return the refund requirements
     */
    public RefundRequirements getRefundRequirements(TransactionMode mode) {
        return service.getRefundCapabilities().getRequirements(mode);
    }

    /**
     * Prepare for submission of a transaction.
     * <p/>
     * This ensures that all the details required to successfully submit the transaction are present.
     *
     * @param transaction the transaction
     * @return the validation status
     * @throws PaymentProcessorException for any error
     */
    public ValidationStatus prepare(Transaction transaction) {
        ValidationStatus result;
        if (transaction instanceof Payment) {
            result = service.prepare((Payment) transaction);
        } else {
            result = service.prepare((Refund) transaction);
        }
        return result;
    }

    /**
     * Submit a transaction.
     *
     * @param transaction the transaction
     * @param isNew   if {@code true}, this is the first attempt at submitting the transaction, otherwise the
     *                transaction has been submitted previously, but the submission failed
     * @throws PaymentProcessorException for any error
     */
    public void submit(Transaction transaction, boolean isNew) {
        if (transaction instanceof Payment) {
            Payment payment = (Payment) transaction;
            service.submit(payment, isNew);
        } else {
            Refund refund = (Refund) transaction;
            service.submit(refund, isNew);
        }
    }

    /**
     * Checks the status of a transaction.
     * <p/>
     * This is used to determine if an {@link Transaction.Status#SUBMITTED SUBMITTED} transaction has been updated.
     *
     * @param transaction the transaction
     * @return {@code true} if the payment was updated, otherwise {@code false}
     * @throws PaymentProcessorException for any error
     */
    public boolean check(Transaction transaction) {
        boolean result;
        if (transaction instanceof Payment) {
            result = service.check((Payment) transaction);
        } else {
            result = service.check((Refund) transaction);
        }
        return result;
    }

    /**
     * Cancel a transaction.
     *
     * @param transaction the transaction to cancel
     * @throws PaymentProcessorException for any error
     */
    public void cancel(Transaction transaction) {
        if (transaction instanceof Payment) {
            service.cancel((Payment) transaction);
        } else {
            service.cancel((Refund) transaction);
        }
    }
}