/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.internal.transaction;

import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.domain.internal.factory.DomainService;
import org.openvpms.paymentprocessor.transaction.Payment;
import org.openvpms.paymentprocessor.transaction.PaymentQuery;
import org.openvpms.paymentprocessor.transaction.Refund;
import org.openvpms.paymentprocessor.transaction.RefundQuery;
import org.openvpms.paymentprocessor.transaction.TransactionQuery;
import org.openvpms.paymentprocessor.transaction.Transactions;

/**
 * Default implementation of {@link Transactions}.
 *
 * @author Tim Anderson
 */
public class TransactionsImpl implements Transactions {

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The transaction factory.
     */
    private final PaymentProcessorTransactionFactory factory;

    /**
     * The domain object service.
     */
    private final DomainService domainService;

    /**
     * Constructs a {@link TransactionsImpl}.
     *
     * @param service       the archetype service
     * @param factory       the transaction factory
     * @param domainService the domain service
     */
    public TransactionsImpl(ArchetypeService service, PaymentProcessorTransactionFactory factory,
                            DomainService domainService) {
        this.service = service;
        this.factory = factory;
        this.domainService = domainService;
    }

    /**
     * Returns a payment given its OpenVPMS identifier.
     *
     * @param id the payment identifier
     * @return the corresponding payment or {@code null} if none is found
     */
    @Override
    public Payment getPayment(long id) {
        return getPaymentQuery()
                .id(id)
                .findFirst();
    }

    /**
     * Returns a refund given its OpenVPMS identifier.
     *
     * @param id the refund identifier
     * @return the corresponding refund or {@code null} if none is found
     */
    @Override
    public Refund getRefund(long id) {
        return getRefundQuery()
                .id(id)
                .findFirst();
    }

    /**
     * Returns a payment, given its payment processor id.
     * <p>
     * A payment can have a single identifier issued by a payment processor. To avoid duplicates, each payment processor
     * must provide a unique archetype.
     *
     * @param archetype the identifier archetype. Must have an <em>actIdentity.paymentProcessor</em> prefix.
     * @param id        the payment identifier
     * @return the corresponding transaction or {@code null} if none is found
     */
    @Override
    public Payment getPayment(String archetype, String id) {
        return getPaymentQuery().transactionId(archetype, id).findFirst();
    }

    /**
     * Returns a refund, given its provider id.
     * <p>
     * A refund can have a single identifier issued by a payment processor. To avoid duplicates, each payment processor
     * must provide a unique archetype.
     *
     * @param archetype the identifier archetype. Must have an <em>actIdentity.paymentProcessor</em> prefix.
     * @param id        the refund identifier
     * @return the corresponding transaction or {@code null} if none is found
     */
    @Override
    public Refund getRefund(String archetype, String id) {
        return getRefundQuery().transactionId(archetype, id).findFirst();
    }

    /**
     * Returns a query to query payments and refunds.
     *
     * @return a new transaction query
     */
    @Override
    public TransactionQuery getQuery() {
        return new TransactionQueryImpl(factory, service, domainService);
    }

    /**
     * Returns a query to query payments.
     *
     * @return a new payment query
     */
    @Override
    public PaymentQuery getPaymentQuery() {
        return new PaymentQueryImpl(factory, service, domainService);
    }

    /**
     * Returns a query to query refunds.
     *
     * @return a new refund query
     */
    @Override
    public RefundQuery getRefundQuery() {
        return new RefundQueryImpl(factory, service, domainService);
    }
}