/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.internal.transaction;

import org.apache.commons.lang3.mutable.Mutable;
import org.apache.commons.lang3.mutable.MutableObject;
import org.openvpms.archetype.rules.finance.account.CustomerAccountRules;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.Identity;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.paymentprocessor.exception.PaymentProcessorException;
import org.openvpms.paymentprocessor.internal.i18n.PaymentProcessorMessages;
import org.openvpms.paymentprocessor.transaction.Transaction.Status;
import org.openvpms.paymentprocessor.transaction.TransactionUpdater;
import org.openvpms.plugin.internal.service.security.RunAsService;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallbackWithoutResult;
import org.springframework.transaction.support.TransactionTemplate;

import java.util.Objects;

/**
 * Default implementation of {@link TransactionUpdater}.
 *
 * @author Tim Anderson
 */
public class TransactionUpdaterImpl implements TransactionUpdater {

    /***
     * The transaction bean.
     */
    private final IMObjectBean bean;

    /**
     * Determines if completing the transaction posts the parent payment/refund.
     */
    private final boolean completionTriggersPost;

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The transaction manager.
     */
    private final PlatformTransactionManager transactionManager;

    /**
     * The customer account rules.
     */
    private final CustomerAccountRules rules;

    /**
     * The run-as service.
     */
    private final RunAsService runAs;

    /**
     * The transaction status.
     */
    private Status status;

    /**
     * The status message.
     */
    private MutableObject<String> message;

    /**
     * The transaction id archetype.
     */
    private String transactionIdArchetype;

    /**
     * The transaction id.
     */
    private String transactionId;

    /**
     * The transaction url.
     */
    private MutableObject<String> url;

    /**
     * Constructs a {@link TransactionUpdaterImpl}.
     *
     * @param bean                   the transaction bean
     * @param completionTriggersPost determines if completing the transaction posts the parent payment/refund
     * @param service                the archetype service
     * @param transactionManager     the transaction manager
     * @param rules                  the customer account rules
     * @param runAs                  the run-as service
     */
    public TransactionUpdaterImpl(IMObjectBean bean, boolean completionTriggersPost, ArchetypeService service,
                                  PlatformTransactionManager transactionManager, CustomerAccountRules rules,
                                  RunAsService runAs) {
        this.bean = bean;
        this.completionTriggersPost = completionTriggersPost;
        this.service = service;
        this.transactionManager = transactionManager;
        this.rules = rules;
        this.runAs = runAs;
    }

    /**
     * Sets the  status.
     *
     * @param status the status
     * @return this
     */
    @Override
    public TransactionUpdater status(Status status) {
        this.status = status;
        return this;
    }

    /**
     * Sets the message.
     *
     * @param message the message
     * @return this
     */
    @Override
    public TransactionUpdater message(String message) {
        this.message = new MutableObject<>(message);
        return this;
    }

    /**
     * Sets the transaction identifier, issued by the provider.
     * <p>
     * A transaction can have a single identifier issued by a provider. To avoid duplicates, each payment processor must
     * provide a unique archetype.
     *
     * @param archetype the identifier archetype. Must have an <em>actIdentity.paymentProcessorTransaction</em> prefix.
     * @param id        the transaction identifier
     * @return this
     */
    @Override
    public TransactionUpdater transactionId(String archetype, String id) {
        transactionIdArchetype = archetype;
        transactionId = id;
        return this;
    }

    /**
     * Sets the customer url for this transaction.
     *
     * @param url the url
     * @return this
     */
    @Override
    public TransactionUpdater url(String url) {
        this.url = new MutableObject<>(url);
        return this;
    }

    /**
     * Commits any changes.
     *
     * @return {@code true} if the transaction was updated, {@code false} if no changes were made
     * @throws PaymentProcessorException if the update fails
     */
    @Override
    public boolean update() {
        Status current = getStatus();
        boolean updated = false;
        boolean completed;
        if (status != null && status != current) {
            if (isTerminalStatus(current) || (current == Status.SUBMITTED && !isTerminalStatus(status))) {
                throw new PaymentProcessorException(PaymentProcessorMessages.cannotChangeStatus(current, status));
            }
            bean.setValue("status", status.toString());
            updated = true;
            completed = status == Status.COMPLETED;
        } else {
            completed = current == Status.COMPLETED;
        }
        updated |= updateIfChanged("message", message);
        updated |= updateTransactionId();
        updated |= updateIfChanged("url", url);

        reset();
        if (updated) {
            bean.save();
        }
        if (completed && completionTriggersPost) {
            post();
        }
        return updated;
    }

    /**
     * Posts the corresponding payment/refund when a transaction is COMPLETED. Note that this requires the transaction
     * to be saved with COMPLETED status prior to invoking {@link CustomerAccountRules#canPost(FinancialAct)}.
     */
    private void post() {
        TransactionTemplate template = new TransactionTemplate(transactionManager);
        template.execute(new TransactionCallbackWithoutResult() {
            @Override
            protected void doInTransactionWithoutResult(TransactionStatus status) {
                FinancialAct item = bean.getSource("transaction", FinancialAct.class);
                if (item != null) {
                    FinancialAct parent = service.getBean(item).getSource("transaction", FinancialAct.class);
                    if (parent != null && rules.canPost(parent)) {
                        runAs.run(() -> rules.post(parent));
                    }
                }
            }
        });
    }

    /**
     * Determines if a status is a terminal one.
     *
     * @param status the status
     * @return {@code true} if the status is terminal, otherwise {@code false}
     */
    private boolean isTerminalStatus(Status status) {
        return status == Status.COMPLETED || status == Status.CANCELLED || status == Status.ERROR;
    }

    /**
     * Updates the transaction identifier.
     *
     * @return {@code true} if the transaction identifier was changed
     */
    private boolean updateTransactionId() {
        boolean updated = false;
        if (transactionId != null) {
            Identity identity = getIdentity(bean, transactionIdArchetype, "transactionId");
            if (!identity.isNew()) {
                if (!identity.isA(transactionIdArchetype)) {
                    throw new PaymentProcessorException(PaymentProcessorMessages.differentTxnIdentifierArchetype(
                            identity.getArchetype(), transactionIdArchetype));
                }
            } else {
                updated = true;
            }
            updated |= updateIdentity(identity, transactionId);
        }
        return updated;
    }

    /**
     * Updates an identity.
     *
     * @param identity the identity to update
     * @param value    the new value
     * @return {@code true} if the identity was updated
     */
    private boolean updateIdentity(Identity identity, String value) {
        boolean updated = false;
        if (!Objects.equals(identity.getIdentity(), value)) {
            identity.setIdentity(value);
            updated = true;
        }
        return updated;
    }

    /**
     * Gets or creates an identity.
     *
     * @param bean      the bean that has the identity
     * @param archetype the identity archetype
     * @param node      the identity node
     * @return the corresponding identity
     */
    private Identity getIdentity(IMObjectBean bean, String archetype, String node) {
        Identity identity = bean.getObject(node, Identity.class);
        if (identity == null) {
            identity = service.create(archetype, Identity.class);
            bean.addValue(node, identity);
        }
        return identity;
    }

    /**
     * Returns the current transaction status.
     *
     * @return the current transaction status. May be {@code null}
     */
    private Status getStatus() {
        Act act = (Act) bean.getObject();
        String result = act.getStatus();
        return result != null ? Status.valueOf(result) : null;
    }

    /**
     * Clears the state.
     */
    private void reset() {
        status = null;
        transactionId = null;
        transactionIdArchetype = null;
        url = null;
    }

    /**
     * Updates a node if its value has changed.
     *
     * @param name  the node name
     * @param value the node value. May be {@code null}
     * @return {@code true} if the node was updated
     */
    private boolean updateIfChanged(String name, Object value) {
        boolean updated = false;
        if (!Objects.equals(value, bean.getValue(name))) {
            bean.setValue(name, value);
            updated = true;
        }
        return updated;
    }

    /**
     * Updates a node if its value has changed.
     *
     * @param name  the node name
     * @param value the node value. May be {@code null}
     * @return {@code true} if the node was updated
     */
    private boolean updateIfChanged(String name, Mutable<?> value) {
        return (value != null) && updateIfChanged(name, value.getValue());
    }
}