/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.internal.transaction;

import org.openvpms.archetype.rules.finance.account.CustomerAccountRules;
import org.openvpms.archetype.rules.practice.PracticeService;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.bean.Policies;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.domain.internal.factory.DomainService;
import org.openvpms.paymentprocessor.transaction.Payment;
import org.openvpms.paymentprocessor.transaction.Refund;
import org.openvpms.plugin.internal.service.security.RunAsService;
import org.springframework.transaction.PlatformTransactionManager;

import java.util.List;

/**
 * Default implementation of {@link Refund}.
 *
 * @author Tim Anderson
 */
public class RefundImpl extends TransactionImpl implements Refund {

    /**
     * Constructs a {@link RefundImpl}.
     *
     * @param act                the act
     * @param service            the archetype service
     * @param domainService      the domain service
     * @param practiceService    the practice service
     * @param rules              the customer account rules
     * @param transactionManager the transaction manager
     * @param runAs              the run-as service
     */
    public RefundImpl(FinancialAct act, ArchetypeService service, DomainService domainService,
                      PracticeService practiceService, CustomerAccountRules rules,
                      PlatformTransactionManager transactionManager, RunAsService runAs) {
        super(act, service, domainService, practiceService, rules, transactionManager, runAs);
    }

    /**
     * Constructs a {@link RefundImpl}.
     *
     * @param bean               a bean wrapping the act
     * @param service            the archetype service
     * @param domainService      the domain object service
     * @param practiceService    the practice service
     * @param rules              the customer account rules
     * @param transactionManager the transaction manager
     * @param runAs              the run-as service
     */
    public RefundImpl(IMObjectBean bean, ArchetypeService service, DomainService domainService,
                      PracticeService practiceService, CustomerAccountRules rules,
                      PlatformTransactionManager transactionManager, RunAsService runAs) {
        super(bean, service, domainService, practiceService, rules, transactionManager, runAs);
    }

    /**
     * Returns the payment that this is a refund for.
     *
     * @return the payment, or {@code null} if this is an unmatched refund
     */
    @Override
    public Payment getPayment() {
        Payment result = null;
        DomainService service = getDomainService();
        // get the act.customerAccountRefundPP
        FinancialAct item = getBean().getSource("transaction", FinancialAct.class);
        if (item != null) {
            // get the corresponding act.customerAccountPaymentPP
            FinancialAct payment = service.getBean(item).getSource("reverses", FinancialAct.class);
            if (payment != null) {
                List<FinancialAct> transactions = service.getBean(payment).getTargets(
                        "transactions", FinancialAct.class, Policies.orderBySequence(false));
                // the transaction with the highest sequence (reversed by the above) will be the COMPLETED
                // act.paymentProcessorPayment
                if (!transactions.isEmpty()) {
                    FinancialAct transaction = transactions.get(0);
                    if (Status.COMPLETED.toString().equals(transaction.getStatus())) {
                        result = service.create(transaction, Payment.class);
                    }
                }
            }
        }
        return result;
    }
}