/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.internal.service;


import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorArchetypes;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.query.criteria.CriteriaBuilder;
import org.openvpms.component.query.criteria.CriteriaQuery;
import org.openvpms.component.query.criteria.Root;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.paymentprocessor.exception.PaymentProcessorException;
import org.openvpms.paymentprocessor.internal.i18n.PaymentProcessorMessages;
import org.openvpms.paymentprocessor.service.PaymentProcessorService;
import org.openvpms.plugin.manager.PluginManager;

import java.util.Collections;
import java.util.List;

/**
 * Payment processor locator.
 *
 * @author Tim Anderson
 */
public class PaymentProcessors {

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The plugin manager.
     */
    private final PluginManager manager;

    /**
     * Constructs an {@link PaymentProcessors}.
     *
     * @param service the archetype service
     * @param manager the plugin manager
     */
    public PaymentProcessors(ArchetypeService service, PluginManager manager) {
        this.manager = manager;
        this.service = service;
    }

    /**
     * Determines if there is at least one service registered.
     *
     * @return {@code true} if there is at least one service registered, otherwise {@code false}
     */
    public boolean paymentProcessorServicesExist() {
        return manager.hasService(PaymentProcessorService.class);
    }

    /**
     * Returns the payment processor configurations.
     *
     * @return the active payment processor configurations
     */
    public List<Entity> getPaymentProcessors() {
        List<Entity> result = Collections.emptyList();
        List<String> archetypes = service.getArchetypes(PaymentProcessorArchetypes.PAYMENT_PROCESSORS, false);
        if (!archetypes.isEmpty()) {
            // only query if archetypes are present, else the query will fail
            CriteriaBuilder builder = service.getCriteriaBuilder();
            CriteriaQuery<Entity> query = builder.createQuery(Entity.class);
            Root<Entity> from = query.from(Entity.class, PaymentProcessorArchetypes.PAYMENT_PROCESSORS);
            query.where(builder.equal(from.get("active"), true));
            result = service.createQuery(query).getResultList();
        }
        return result;
    }

    /**
     * Returns the payment processor given its configuration.
     *
     * @param config the terminal
     * @return the corresponding payment processor
     * @throws PaymentProcessorException if the service is unavailable
     */
    public PaymentProcessorService getPaymentProcessor(Entity config) {
        PaymentProcessorService result = null;
        List<PaymentProcessorService> services = manager.getServices(PaymentProcessorService.class);
        String archetype = config.getArchetype();
        for (PaymentProcessorService service : services) {
            if (archetype.equals(service.getArchetype())) {
                result = service;
                break;
            }
        }
        if (result == null) {
            throw new PaymentProcessorException(PaymentProcessorMessages.serviceUnavailable(config.getName()));
        }
        return result;
    }
}
