/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.internal.transaction;


import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.archetype.rules.finance.account.CustomerAccountRules;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorArchetypes;
import org.openvpms.archetype.rules.math.Currency;
import org.openvpms.archetype.rules.practice.PracticeService;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.component.business.domain.im.common.IMObjectReference;
import org.openvpms.component.business.service.archetype.helper.TypeHelper;
import org.openvpms.component.model.act.ActIdentity;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.entity.EntityIdentity;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.domain.customer.Customer;
import org.openvpms.domain.internal.factory.DomainService;
import org.openvpms.domain.practice.Location;
import org.openvpms.paymentprocessor.exception.PaymentProcessorException;
import org.openvpms.paymentprocessor.processor.PaymentProcessor;
import org.openvpms.paymentprocessor.processor.TransactionMode;
import org.openvpms.paymentprocessor.service.TransactionRequirements;
import org.openvpms.paymentprocessor.transaction.Transaction;
import org.openvpms.paymentprocessor.transaction.TransactionUpdater;
import org.openvpms.plugin.internal.service.security.RunAsService;
import org.springframework.transaction.PlatformTransactionManager;

import java.math.BigDecimal;
import java.time.OffsetDateTime;

/**
 * Default implementation of {@link Transaction}.
 *
 * @author Tim Anderson
 */
public class TransactionImpl implements Transaction {

    /**
     * The transaction act.
     */
    private final FinancialAct act;

    /**
     * Bean wrapping the act.
     */
    private final IMObjectBean bean;

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The domain object service.
     */
    private final DomainService domainService;

    /**
     * The practice service.
     */
    private final PracticeService practiceService;

    /**
     * The customer account rules.
     */
    private final CustomerAccountRules rules;

    /**
     * The transaction manager.
     */
    private final PlatformTransactionManager transactionManager;

    /**
     * The run-as service.
     */
    private final RunAsService runAs;

    /**
     * Cached customer.
     */
    private Customer customer;

    /**
     * Determines if completing the transaction posts the parent payment or refund.
     */
    private boolean completionTriggersPost = false;

    /**
     * Transaction id node.
     */
    static final String TRANSACTION_ID = "transactionId";

    /**
     * Constructs a {@link TransactionImpl}.
     *
     * @param act                the act
     * @param service            the archetype service
     * @param domainService      the domain object service
     * @param practiceService    the practice service
     * @param rules              the customer account rules
     * @param transactionManager the transaction manager
     * @param runAs              the run-as service
     */
    public TransactionImpl(FinancialAct act, ArchetypeService service, DomainService domainService,
                           PracticeService practiceService, CustomerAccountRules rules,
                           PlatformTransactionManager transactionManager, RunAsService runAs) {
        this.act = act;
        this.service = service;
        this.bean = service.getBean(act);
        this.domainService = domainService;
        this.practiceService = practiceService;
        this.rules = rules;
        this.transactionManager = transactionManager;
        this.runAs = runAs;
    }

    /**
     * Constructs a {@link TransactionImpl}.
     *
     * @param bean               a bean wrapping the act
     * @param service            the archetype service
     * @param domainService      the domain object service
     * @param practiceService    the practice service
     * @param rules              the customer account rules
     * @param transactionManager the transaction manager
     * @param runAs              the run-as service
     */
    public TransactionImpl(IMObjectBean bean, ArchetypeService service, DomainService domainService,
                           PracticeService practiceService, CustomerAccountRules rules,
                           PlatformTransactionManager transactionManager, RunAsService runAs) {
        this.act = bean.getObject(FinancialAct.class);
        this.service = service;
        this.bean = bean;
        this.domainService = domainService;
        this.practiceService = practiceService;
        this.rules = rules;
        this.transactionManager = transactionManager;
        this.runAs = runAs;
    }

    /**
     * Returns the OpenVPMS identifier for the transaction.
     *
     * @return the identifier
     */
    @Override
    public long getId() {
        return act.getId();
    }

    /**
     * Returns the mode to use for this transaction.
     *
     * @return the transaction mode
     */
    @Override
    public TransactionMode getTransactionMode() {
        return TransactionMode.valueOf(getBean().getString("transactionMode"));
    }

    /**
     * Returns the reference of the parent payment or refund.
     *
     * @return the parent payment or refund reference
     */
    @Override
    public Reference getParent() {
        ActIdentity identity = bean.getObject("parentId", ActIdentity.class);
        long id = identity != null ? Long.parseLong(identity.getIdentity()) : -1;
        String archetype = act.isA(PaymentProcessorArchetypes.PAYMENT) ? CustomerAccountArchetypes.PAYMENT
                                                                       : CustomerAccountArchetypes.REFUND;
        return new IMObjectReference(archetype, id);
    }

    /**
     * Indicates whether some other object is "equal to" this one.
     *
     * @param obj the reference object with which to compare.
     * @return {@code true} if this object is the same as the obj
     * argument; {@code false} otherwise.
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        } else if (obj instanceof TransactionImpl) {
            return act.equals(((TransactionImpl) obj).act);
        }
        return false;
    }

    /**
     * Returns a hash code value for the object.
     *
     * @return a hash code value for this object.
     */
    @Override
    public int hashCode() {
        return act.hashCode();
    }

    /**
     * Returns the transaction identifier, issued by the payment processor.
     *
     * @return the transaction identifier, or {@code null} if none has been issued
     */
    @Override
    public String getTransactionId() {
        ActIdentity identity = getTransactionIdentity();
        return identity != null ? identity.getIdentity() : null;
    }

    /**
     * Returns the identity for the transaction, issued by the payment processor.
     *
     * @return the transaction identifier, or {@code null} if none has been issued
     */
    @Override
    public ActIdentity getTransactionIdentity() {
        return bean.getObject(TRANSACTION_ID, ActIdentity.class);
    }

    /**
     * Returns the transaction date.
     *
     * @return the transaction date
     */
    @Override
    public OffsetDateTime getDate() {
        return DateRules.toOffsetDateTime(act.getActivityStartTime());
    }

    /**
     * Returns the transaction status.
     *
     * @return the transaction status
     */
    @Override
    public Status getStatus() {
        return Status.valueOf(act.getStatus());
    }

    /**
     * Returns the message.
     *
     * @return the message. May be {@code null}
     */
    @Override
    public String getMessage() {
        return bean.getString("message");
    }

    /**
     * Sets the status.
     *
     * @param status the status
     * @throws PaymentProcessorException if the status cannot be updated
     */
    @Override
    public void setStatus(Status status) {
        state().status(status)
                .update();
    }

    /**
     * Sets the status and message.
     *
     * @param status  the status
     * @param message the status message. May be {@code null}
     * @throws PaymentProcessorException if the status cannot be updated
     */
    @Override
    public void setStatus(Status status, String message) {
        state().status(status)
                .message(message)
                .update();
    }

    /**
     * Returns the customer the transaction is for.
     *
     * @return the customer
     */
    @Override
    public Customer getCustomer() {
        if (customer == null) {
            Party party = bean.getTarget("customer", Party.class);
            if (party == null) {
                throw new IllegalStateException("Transaction has no customer");
            }
            customer = domainService.create(party, Customer.class);
        }
        return customer;
    }

    /**
     * Returns the identity for the customer, issued by the payment processor.
     * <p/>
     * This is short for {@code getCustomerIdentity(archetype).getIdentity()}.
     *
     * @param archetype the identity archetype. Must have a <em>entityIdentity.paymentProcessor</em> prefix.
     * @return the corresponding identity, or {@code null} if none exists
     */
    @Override
    public String getCustomerId(String archetype) {
        EntityIdentity identity = getCustomerIdentity(archetype);
        return identity != null ? identity.getIdentity() : null;
    }

    /**
     * Returns the identity for the customer, issued by the payment processor.
     *
     * @param archetype the identity archetype. Must have a <em>entityIdentity.paymentProcessor</em> prefix.
     * @return the customer identifier, or {@code null} if none has been issued
     */
    @Override
    public EntityIdentity getCustomerIdentity(String archetype) {
        if (!TypeHelper.isA(archetype, "entityIdentity.paymentProcessor*")) {
            throw new IllegalArgumentException("Invalid identity archetype: " + archetype);
        }
        return getCustomer().getIdentity(archetype);
    }

    /**
     * Returns the payment processor to use.
     *
     * @return the payment processor
     */
    @Override
    public PaymentProcessor getPaymentProcessor() {
        Entity entity = bean.getTarget("paymentProcessor", Entity.class);
        if (entity == null) {
            throw new IllegalStateException("Transaction has no payment processor");
        }
        return domainService.create(entity, PaymentProcessor.class);
    }

    /**
     * Returns the practice location.
     *
     * @return the practice location
     */
    @Override
    public Location getLocation() {
        Party location = bean.getTarget("location", Party.class);
        if (location == null) {
            throw new IllegalStateException("Transaction has no location");
        }
        return domainService.create(location, Location.class);
    }

    /**
     * Returns the transaction amount.
     *
     * @return the transaction amount
     */
    @Override
    public BigDecimal getAmount() {
        return bean.getBigDecimal("amount", BigDecimal.ZERO);
    }

    /**
     * Returns the transaction currency code.
     * <p/>
     * This is a 3 character code as specified by ISO 4217.
     *
     * @return the transaction currency code
     */
    @Override
    public String getCurrencyCode() {
        Currency currency = practiceService.getCurrency();
        if (currency == null) {
            throw new IllegalStateException("No practice currency");
        }
        return currency.getCode();
    }

    /**
     * Returns the transaction currency.
     *
     * @return the transaction currency
     */
    @Override
    public java.util.Currency getCurrency() {
        String code = getCurrencyCode();
        java.util.Currency result = java.util.Currency.getInstance(code);
        if (result == null) {
            throw new IllegalStateException("Failed to determine currency for currency code=" + code);
        }
        return result;
    }

    /**
     * Returns the customer email to use for this transaction, for notification purposes.
     * <p/>
     * This is only populated if the {@link TransactionRequirements} indicate that it is optional or required.
     *
     * @return the customer email. May be {@code null}
     */
    @Override
    public String getEmail() {
        return getBean().getString("email");
    }

    /**
     * Determines if the customer has been notified of the transaction by the payment processor.
     *
     * @return {@code true} if the customer has been notified, otherwise {@code false}
     */
    @Override
    public boolean getNotified() {
        return getBean().getBoolean("notified");
    }

    /**
     * Returns the customer URL for this transaction.
     *
     * @return the customer URL. May be {@code null}
     */
    @Override
    public String getUrl() {
        return getBean().getString("url");
    }

    /**
     * Returns an updater to change the state of the transaction.
     *
     * @return the updater
     */
    @Override
    public TransactionUpdater state() {
        return new TransactionUpdaterImpl(bean, completionTriggersPost, service, transactionManager, rules, runAs);
    }

    /**
     * Determines if completing the transaction posts the parent payment/refund.
     * <p/>
     * Posting only occurs if all transactions are complete.
     * <p/>
     * Defaults to {@code false}.
     *
     * @return {@code true}, completion triggers post
     */
    public boolean completionTriggersPost() {
        return completionTriggersPost;
    }

    /**
     * Determines if completing the transaction posts the parent payment/refund.
     * <p/>
     * Posting only occurs if all transactions are complete.
     *
     * @param completionTriggersPost if {@code true}, completion triggers post
     */
    public void setCompletionTriggersPost(boolean completionTriggersPost) {
        this.completionTriggersPost = completionTriggersPost;
    }

    /**
     * Returns the bean wrapping the act.
     *
     * @return the bean
     */
    protected IMObjectBean getBean() {
        return bean;
    }

    /**
     * Returns the domain object service.
     *
     * @return the domain object service
     */
    protected DomainService getDomainService() {
        return domainService;
    }
}