/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.internal.transaction;

import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorArchetypes;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorTransactionStatus;
import org.openvpms.component.model.act.ActIdentity;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.domain.internal.factory.DomainService;
import org.openvpms.paymentprocessor.processor.TransactionMode;
import org.openvpms.paymentprocessor.transaction.Transaction;

import java.math.BigDecimal;

/**
 * Factory for payment processor transactions.
 *
 * @author Tim Anderson
 */
public class PaymentProcessorTransactionFactory {

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The domain object service.
     */
    private final DomainService domainService;

    /**
     * Constructs an {@link PaymentProcessorTransactionFactory}.
     *
     * @param service       the archetype service
     * @param domainService the domain object service
     */
    public PaymentProcessorTransactionFactory(ArchetypeService service, DomainService domainService) {
        this.service = service;
        this.domainService = domainService;
    }

    /**
     * Creates a new <em>act.paymentProcessorPayment</em>,
     * with {@link PaymentProcessorTransactionStatus#PENDING PENDING} status.
     *
     * @param parentPayment    the parent customer payment. Must be persistent, and have populated customer and location
     * @param amount           the payment amount
     * @param paymentProcessor the payment processor
     * @param mode             the transaction mode
     * @param email            the customer email address. May be {@code null}
     * @param description      the transaction description. May be {@code null}
     * @return a new payment
     */
    public FinancialAct createPayment(FinancialAct parentPayment, BigDecimal amount, Entity paymentProcessor,
                                      TransactionMode mode, String email, String description) {
        return createTransaction(parentPayment, CustomerAccountArchetypes.PAYMENT, PaymentProcessorArchetypes.PAYMENT,
                                 amount, paymentProcessor, mode, email, description);
    }

    /**
     * Creates a new <em>act.paymentProcessorRefund</em>,
     * with {@link PaymentProcessorTransactionStatus#PENDING PENDING} status.
     *
     * @param parentRefund     the parent customer refund. Must be persistent, and have populated customer and location
     * @param amount           the refund amount
     * @param paymentProcessor the payment processor
     * @param mode             the transaction mode
     * @param email            the customer email address. May be {@code null}
     * @param description      the transaction description. May be {@code null}
     * @return a new refund
     */
    public FinancialAct createRefund(FinancialAct parentRefund, BigDecimal amount, Entity paymentProcessor,
                                     TransactionMode mode, String email, String description) {
        return createTransaction(parentRefund, CustomerAccountArchetypes.REFUND, PaymentProcessorArchetypes.REFUND,
                                 amount, paymentProcessor, mode, email, description);
    }

    /**
     * Returns a transaction for an <em>act.paymentProcessorPayment</em> or <em>act.paymentProcessorRefund</em>.
     *
     * @param act the act
     * @return the corresponding transaction
     */
    public Transaction getTransaction(FinancialAct act) {
        return getTransaction(act, false);
    }

    /**
     * Returns a transaction for an <em>act.paymentProcessorPayment</em> or <em>act.paymentProcessorRefund</em>.
     *
     * @param act                    the act
     * @param completionTriggersPost if {@code true}, setting the status to COMPLETED posts the parent payment or refund
     * @return the corresponding transaction
     */
    public Transaction getTransaction(FinancialAct act, boolean completionTriggersPost) {
        TransactionImpl transaction;
        if (act.isA(PaymentProcessorArchetypes.PAYMENT)) {
            transaction = domainService.create(act, PaymentImpl.class);
        } else if (act.isA(PaymentProcessorArchetypes.REFUND)) {
            transaction = domainService.create(act, RefundImpl.class);
        } else {
            throw new IllegalArgumentException("Invalid transaction type: " + act.getArchetype()
                                               + " for argument 'act'");
        }
        transaction.setCompletionTriggersPost(completionTriggersPost);
        return transaction;
    }

    /**
     * Creates a new payment processor transaction.
     *
     * @param parent           the parent customer act, used for reconciliation purposes
     * @param parentArchetype  the expected parent archetype
     * @param archetype        the transaction archetype
     * @param paymentProcessor the payment processor terminal
     * @param mode             the transaction mode
     * @param email            the customer email address. May be {@code null}
     * @param description      the transaction description. May be {@code null}
     * @return a new transaction
     */
    private FinancialAct createTransaction(FinancialAct parent, String parentArchetype, String archetype,
                                           BigDecimal amount, Entity paymentProcessor, TransactionMode mode,
                                           String email, String description) {
        if (parent.isNew()) {
            throw new IllegalStateException("Parent act must be saved");
        }
        if (!parent.isA(parentArchetype)) {
            throw new IllegalArgumentException("Expected parent of type " + parentArchetype + " but got "
                                               + parent.getArchetype());
        }
        IMObjectBean parentBean = service.getBean(parent);
        Reference customer = parentBean.getTargetRef("customer");
        if (customer == null) {
            throw new IllegalStateException("Parent act has no customer");
        }
        Reference location = parentBean.getTargetRef("location");
        if (location == null) {
            throw new IllegalStateException("Parent act has no location");
        }

        FinancialAct act = service.create(archetype, FinancialAct.class);
        ActIdentity identity = service.create(PaymentProcessorArchetypes.PARENT_ID, ActIdentity.class);
        identity.setIdentity(Long.toString(parent.getId()));
        act.addIdentity(identity);

        IMObjectBean bean = service.getBean(act);
        bean.setValue("amount", amount);
        bean.setTarget("customer", customer);
        bean.setValue("email", email);
        bean.setTarget("paymentProcessor", paymentProcessor);
        bean.setValue("transactionMode", mode.toString());
        bean.setTarget("location", location);
        act.setDescription(description);
        act.setStatus(PaymentProcessorTransactionStatus.PENDING);
        return act;
    }
}