/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.internal.transaction;

import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorArchetypes;
import org.openvpms.component.business.service.archetype.helper.TypeHelper;
import org.openvpms.component.model.act.ActIdentity;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.query.criteria.CriteriaBuilder;
import org.openvpms.component.query.criteria.CriteriaQuery;
import org.openvpms.component.query.criteria.Join;
import org.openvpms.component.query.criteria.Root;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.domain.internal.factory.DomainService;
import org.openvpms.domain.internal.query.DomainQueryImpl;
import org.openvpms.domain.query.Filter;
import org.openvpms.paymentprocessor.processor.PaymentProcessor;
import org.openvpms.paymentprocessor.transaction.AbstractTransactionQuery;
import org.openvpms.paymentprocessor.transaction.Transaction;
import org.openvpms.paymentprocessor.transaction.Transaction.Status;

import javax.persistence.criteria.Predicate;
import java.util.List;

/**
 * Default implementation of {@link AbstractTransactionQuery}.
 *
 * @author Tim Anderson
 */
public class AbstractTransactionQueryImpl<T extends Transaction, Q extends AbstractTransactionQuery<T, Q>>
        extends DomainQueryImpl<T, FinancialAct, Q> implements AbstractTransactionQuery<T, Q> {

    /**
     * The payment processor transaction factory.
     */
    private final PaymentProcessorTransactionFactory factory;

    /**
     * The transaction id archetype.
     */
    private String idArchetype;

    /**
     * The transaction id.
     */
    private String id;

    /**
     * The payment processor.
     */
    private PaymentProcessor processor;

    /**
     * The transaction status.
     */
    private Status status;

    /**
     * Constructs an {@link AbstractTransactionQueryImpl}.
     *
     * @param domainTpe     the domain type
     * @param service       the service
     * @param domainService the domain service
     */
    protected AbstractTransactionQueryImpl(Class<T> domainTpe, PaymentProcessorTransactionFactory factory,
                                           ArchetypeService service, DomainService domainService) {
        super(domainTpe, FinancialAct.class, service, domainService);
        this.factory = factory;
    }

    /**
     * Query transactions that have transaction identifiers with the specified archetype.
     *
     * @param archetype the identifier archetype. Must have an <em>actIdentity.paymentProcessorTransaction</em> prefix.
     * @return this
     */
    @Override
    public Q transactionId(String archetype) {
        return transactionId(archetype, null);
    }

    /**
     * Query transactions with the specified transaction identifier.
     *
     * @param archetype the identifier archetype. Must have an <em>actIdentity.paymentProcessorTransaction</em> prefix.
     * @param id        the transaction identifier
     * @return this
     */
    @Override
    public Q transactionId(String archetype, String id) {
        if (!TypeHelper.matches(archetype, PaymentProcessorArchetypes.TRANSACTION_IDS)) {
            throw new IllegalArgumentException("Invalid transaction id archetype: " + archetype);
        }
        this.idArchetype = archetype;
        this.id = id;
        return getThis();
    }

    /**
     * Query transactions with the specified payment processor.
     *
     * @param processor the payment processor
     * @return this
     */
    @Override
    public Q paymentProcessor(PaymentProcessor processor) {
        this.processor = processor;
        return getThis();
    }

    /**
     * Query transactions with the specified status.
     *
     * @param status the status
     * @return this
     */
    @Override
    public Q status(Status status) {
        this.status = status;
        return getThis();
    }

    /**
     * Adds predicates.
     *
     * @param predicates collects the predicates
     * @param query      the query
     * @param from       the from clause
     * @param builder    the criteria builder
     */
    @Override
    protected void addPredicates(List<Predicate> predicates, CriteriaQuery<FinancialAct> query, Root<FinancialAct> from,
                                 CriteriaBuilder builder) {
        super.addPredicates(predicates, query, from, builder);
        if (idArchetype != null) {
            Join<FinancialAct, ActIdentity> transactionIds = from.join("transactionId", idArchetype);
            transactionIds.alias("transactionIds");
            if (id != null) {
                predicates.add(createPredicate(transactionIds.get("identity"), Filter.equal(id), builder));
            }
        }
        if (processor != null) {
            Join<FinancialAct, Entity> processors = from.join("paymentProcessor");
            processors.alias("processors").on(builder.equal(processors.get("entity"), processor.getObjectReference()));
        }
        if (status != null) {
            predicates.add(createPredicate(from.get("status"), Filter.equal(status.toString()), builder));
        }
    }

    /**
     * Creates a domain object for the given model object.
     *
     * @param object the model object
     * @return the domain object
     */
    @Override
    @SuppressWarnings("unchecked")
    protected T createDomainObject(FinancialAct object) {
        return (T) factory.getTransaction(object, true);
    }
}