/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.transaction;

import org.openvpms.component.model.act.ActIdentity;
import org.openvpms.component.model.entity.EntityIdentity;
import org.openvpms.component.model.object.Reference;
import org.openvpms.domain.customer.Customer;
import org.openvpms.domain.practice.Location;
import org.openvpms.paymentprocessor.exception.PaymentProcessorException;
import org.openvpms.paymentprocessor.processor.PaymentProcessor;
import org.openvpms.paymentprocessor.processor.TransactionMode;
import org.openvpms.paymentprocessor.service.TransactionRequirements;

import java.math.BigDecimal;
import java.time.OffsetDateTime;
import java.util.Currency;

/**
 * Payment processor transaction.
 *
 * @author Tim Anderson
 */
public interface Transaction {

    enum Status {
        PENDING,      // transaction hasn't been submitted
        IN_PROGRESS,  // transaction is prepared for submission
        SUBMITTED,    // transaction has been submitted
        COMPLETED,    // transaction has been paid/refunded
        CANCELLED,    // transaction has been cancelled
        ERROR         // transaction failed with an unrecoverable error
    }

    /**
     * Returns the OpenVPMS identifier for the transaction.
     *
     * @return the identifier
     */
    long getId();

    /**
     * Returns the mode to use for this transaction.
     *
     * @return the transaction mode
     */
    TransactionMode getTransactionMode();

    /**
     * Returns the reference of the parent payment or refund.
     *
     * @return the parent payment or refund reference
     */
    Reference getParent();

    /**
     * Returns the transaction identifier, issued by the payment processor.
     * <p/>
     * This is short for {@code getTransactionIdentity().getIdentity()}.
     *
     * @return the transaction identifier, or {@code null} if none has been issued
     */
    String getTransactionId();

    /**
     * Returns the identity for the transaction, issued by the payment processor.
     *
     * @return the transaction identifier, or {@code null} if none has been issued
     */
    ActIdentity getTransactionIdentity();

    /**
     * Returns the transaction date.
     *
     * @return the transaction date
     */
    OffsetDateTime getDate();

    /**
     * Returns the transaction status.
     *
     * @return the transaction status
     */
    Status getStatus();

    /**
     * Returns the message.
     *
     * @return the message. May be {@code null}
     */
    String getMessage();

    /**
     * Sets the status.
     *
     * @param status the status
     * @throws PaymentProcessorException if the status cannot be updated
     */
    void setStatus(Status status);

    /**
     * Sets the status and message.
     *
     * @param status  the status
     * @param message the status message. May be {@code null}
     * @throws PaymentProcessorException if the status cannot be updated
     */
    void setStatus(Status status, String message);

    /**
     * Returns the customer the transaction is for.
     *
     * @return the customer
     */
    Customer getCustomer();

    /**
     * Returns the identity for the customer, issued by the payment processor.
     * <p/>
     * This is short for {@code getCustomerIdentity(archetype).getIdentity()}.
     *
     * @param archetype the identity archetype. Must have a <em>entityIdentity.paymentProcessor</em> prefix.
     * @return the corresponding identity, or {@code null} if none exists
     */
    String getCustomerId(String archetype);

    /**
     * Returns the identity for the customer, issued by the payment processor.
     *
     * @param archetype the identity archetype. Must have a <em>entityIdentity.paymentProcessor</em> prefix.
     * @return the customer identifier, or {@code null} if none has been issued
     */
    EntityIdentity getCustomerIdentity(String archetype);

    /**
     * Returns the payment processor to use.
     *
     * @return the payment processor
     */
    PaymentProcessor getPaymentProcessor();

    /**
     * Returns the practice location.
     *
     * @return the practice location
     */
    Location getLocation();

    /**
     * Returns the transaction amount.
     *
     * @return the transaction amount
     */
    BigDecimal getAmount();

    /**
     * Returns the transaction currency code.
     * <p/>
     * This is a 3 character code as specified by ISO 4217.
     *
     * @return the transaction currency code
     */
    String getCurrencyCode();

    /**
     * Returns the transaction currency.
     *
     * @return the transaction currency
     */
    Currency getCurrency();

    /**
     * Returns the customer email to use for this transaction, for notification purposes.
     * <p/>
     * This is only populated if the {@link TransactionRequirements} indicate that it is optional or required.
     *
     * @return the customer email. May be {@code null}
     */
    String getEmail();

    /**
     * Determines if the customer has been notified of the transaction by the payment processor.
     *
     * @return {@code true} if the customer has been notified, otherwise {@code false}
     */
    boolean getNotified();

    /**
     * Returns the customer URL for this transaction.
     *
     * @return the customer URL. May be {@code null}
     */
    String getUrl();

    /**
     * Returns an updater to change the state of the transaction.
     * <p/>
     * Use this when updating multiple attributes.
     *
     * @return the updater
     */
    TransactionUpdater state();

}