/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.service;

import org.openvpms.paymentprocessor.service.TransactionRequirements.Field;

/**
 * {@link TransactionRequirements} builder.
 *
 * @author Tim Anderson
 */
public abstract class TransactionRequirementsBuilder<T extends TransactionRequirementsBuilder<T>> {

    /**
     * Determines if the description field is required or not.
     */
    private Field description = Field.OPTIONAL;

    /**
     * Determines if the email field is required or not.
     */
    private Field email = Field.OPTIONAL;

    /**
     * Determines if the description field is required.
     *
     * @param requirement the field requirement
     * @return this
     */
    @SuppressWarnings("unchecked")
    public T description(Field requirement) {
        this.description = requirement;
        return (T) this;
    }

    /**
     * Returns the description field requirement.
     *
     * @return the field requirement
     */
    public Field getDescription() {
        return description;
    }

    /**
     * Determines if the customer email field is required.
     *
     * @param requirement the field requirement
     * @return this
     */
    @SuppressWarnings("unchecked")
    public T email(Field requirement) {
        this.email = requirement;
        return (T) this;
    }

    /**
     * Returns the customer email field requirement.
     *
     * @return this
     */
    public Field getEmail() {
        return email;
    }

    /**
     * Builds the transaction requirements.
     *
     * @return the transaction requirements
     */
    public abstract TransactionRequirements build();

    /**
     * Default implementation of {@link TransactionRequirements}.
     */
    protected abstract static class DefaultTransactionRequirements implements TransactionRequirements {

        /**
         * The description field requirement.
         */
        private final Field description;

        /**
         * The email field requirement.
         */
        private final Field email;

        /**
         * Constructs a {@link DefaultTransactionRequirements}.
         *
         * @param builder the builder to populate this from
         */
        protected DefaultTransactionRequirements(TransactionRequirementsBuilder<?> builder) {
            this.description = builder.getDescription();
            this.email = builder.getEmail();
        }

        /**
         * Determines if the transaction description is required.
         *
         * @return the description requirement
         */
        @Override
        public Field getDescription() {
            return description;
        }

        /**
         * Determines if the customer email is required.
         *
         * @return the email requirement
         */
        @Override
        public Field getEmail() {
            return email;
        }
    }
}