/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.service;

import org.openvpms.paymentprocessor.processor.TransactionMode;

/**
 * Payment processor service transaction mode builder.
 *
 * @author Tim Anderson
 */
public class TransactionModeBuilder<T extends TransactionRequirements, B extends TransactionCapabilitiesBuilder<T, B>> {

    /**
     * The parent builder.
     */
    private final B parent;

    /**
     * The transaction mode.
     */
    private final TransactionMode mode;

    /**
     * The requirements.
     */
    private T requirements;

    /**
     * Constructs a {@link TransactionModeBuilder}.
     *
     * @param parent the parent builder
     * @param mode   the transaction mode
     */
    protected TransactionModeBuilder(B parent, TransactionMode mode) {
        this.parent = parent;
        this.mode = mode;
    }

    /**
     * Sets the requirements.
     *
     * @param requirements the requirements
     * @return this
     */
    public TransactionModeBuilder<T, B> requirements(T requirements) {
        this.requirements = requirements;
        return this;
    }

    /**
     * Adds the transaction mode.
     *
     * @return the parent builder
     */
    public B add() {
        if (requirements == null) {
            throw new IllegalStateException("No transaction requirements specified");
        }
        return parent.add(new TransactionCapabilitiesBuilder.TransactionModeCapability<>(mode, requirements));
    }
}
