/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.service;

/**
 * Transaction validation status.
 *
 * @author Tim Anderson
 */
public class ValidationStatus {

    public enum Status {
        VALID,                // transaction is valid and may be submitted
        WARNING,              // the transaction may be submitted, but may be rejected. Details are provided in the
                              // message.
        ERROR                 // transaction is invalid and may not be submitted. Details are provided in the message.
    }

    /**
     * The status.
     */
    private final Status status;

    /**
     * The status message, for warnings and errors.
     */
    private final String message;

    /**
     * Construct a {@link ValidationStatus}.
     *
     * @param status the status
     */
    private ValidationStatus(Status status) {
        this(status, null);
    }

    /**
     * Construct a {@link ValidationStatus}.
     *
     * @param status  the status
     * @param message the message
     */
    private ValidationStatus(Status status, String message) {
        this.status = status;
        this.message = message;
    }

    /**
     * Returns the status.
     *
     * @return the status
     */
    public Status getStatus() {
        return status;
    }

    /**
     * Returns the message.
     *
     * @return the message. May be {@code null} if the status is {@code VALID}
     */
    public String getMessage() {
        return message;
    }

    /**
     * Creates a status indicating that the claim is valid.
     *
     * @return a new status
     */
    public static ValidationStatus valid() {
        return new ValidationStatus(Status.VALID);
    }

    /**
     * Creates a status indicating that the claim is valid but has a warning.
     *
     * @param message the warning message
     * @return a new status
     */
    public static ValidationStatus warning(String message) {
        if (message == null) {
            throw new IllegalArgumentException("Argument 'message' must be provided");
        }
        return new ValidationStatus(Status.WARNING, message);
    }

    /**
     * Creates a status indicating that the claim is invalid..
     *
     * @param message the error message
     * @return a new status
     */
    public static ValidationStatus error(String message) {
        if (message == null) {
            throw new IllegalArgumentException("Argument 'message' must be provided");
        }
        return new ValidationStatus(Status.ERROR, message);
    }
}
