/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.transaction;

/**
 * Transaction service.
 *
 * @author Tim Anderson
 */
public interface Transactions {

    /**
     * Returns a payment given its OpenVPMS identifier.
     *
     * @param id the payment identifier
     * @return the corresponding payment or {@code null} if none is found
     */
    Payment getPayment(long id);

    /**
     * Returns a refund given its OpenVPMS identifier.
     *
     * @param id the refund identifier
     * @return the corresponding refund or {@code null} if none is found
     */
    Refund getRefund(long id);

    /**
     * Returns a payment, given its payment processor id.
     * <p>
     * A payment can have a single identifier issued by a payment processor. To avoid duplicates, each payment processor
     * must provide a unique archetype.
     *
     * @param archetype the identifier archetype. Must have an <em>actIdentity.paymentProcessor</em> prefix.
     * @param id        the payment identifier
     * @return the corresponding transaction or {@code null} if none is found
     */
    Payment getPayment(String archetype, String id);

    /**
     * Returns a refund, given its provider id.
     * <p>
     * A refund can have a single identifier issued by a payment processor. To avoid duplicates, each payment processor
     * must provide a unique archetype.
     *
     * @param archetype the identifier archetype. Must have an <em>actIdentity.paymentProcessor</em> prefix.
     * @param id        the refund identifier
     * @return the corresponding transaction or {@code null} if none is found
     */
    Refund getRefund(String archetype, String id);

    /**
     * Returns a query to query payments and refunds.
     *
     * @return a new transaction query
     */
    TransactionQuery getQuery();

    /**
     * Returns a query to query payments.
     *
     * @return a new payment query
     */
    PaymentQuery getPaymentQuery();

    /**
     * Returns a query to query refunds.
     *
     * @return a new refund query
     */
    RefundQuery getRefundQuery();

}
