/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.service;

import org.openvpms.paymentprocessor.processor.TransactionMode;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

/**
 * {@link TransactionCapabilities} builder.
 *
 * @author Tim Anderson
 */
public abstract class TransactionCapabilitiesBuilder<T extends TransactionRequirements,
        B extends TransactionCapabilitiesBuilder<T, B>> {

    /**
     * The transaction mode capabilities, keyed on transaction mode.
     */
    private final Map<TransactionMode, TransactionModeCapability<T>> capabilities = new LinkedHashMap<>();

    /**
     * Adds a transaction mode and their requirements.
     *
     * @param mode         the transaction mode
     * @param requirements the requirements
     * @return this
     */
    public B add(TransactionMode mode, T requirements) {
        return mode(mode).requirements(requirements).add();
    }

    /**
     * Returns a builder for a transaction mode.
     *
     * @param mode the transaction mode
     * @return a new builder
     */
    public TransactionModeBuilder<T, B> mode(TransactionMode mode) {
        return new TransactionModeBuilder<>(getThis(), mode);
    }

    /**
     * Builds the capabilities.
     *
     * @return the capabilities
     */
    public abstract TransactionCapabilities<T> build();

    /**
     * Adds a capability.
     *
     * @param capability the capability
     * @return this
     */
    protected B add(TransactionModeCapability<T> capability) {
        capabilities.put(capability.getMode(), capability);
        return getThis();
    }

    protected Map<TransactionMode, TransactionModeCapability<T>> getCapabilities() {
        return capabilities;
    }

    @SuppressWarnings("unchecked")
    private B getThis() {
        return (B) this;
    }

    protected static class DefaultTransactionCapabilities<T extends TransactionRequirements>
            implements TransactionCapabilities<T> {

        private final Map<TransactionMode, TransactionModeCapability<T>> capabilities;

        public DefaultTransactionCapabilities(Map<TransactionMode, TransactionModeCapability<T>> capabilities) {
            this.capabilities = capabilities;
        }

        /**
         * Returns the transaction modes that this supports.
         *
         * @return the transaction modes
         */
        @Override
        public Set<TransactionMode> getTransactionModes() {
            return capabilities.keySet();
        }

        /**
         * Returns the requirements for the specified transaction mode.
         *
         * @param mode the transaction mode
         * @return the requirements
         */
        @Override
        public T getRequirements(TransactionMode mode) {
            TransactionModeCapability<T> capability = capabilities.get(mode);
            return capability != null ? capability.getRequirements() : null;
        }
    }

    /**
     * Payment processor transaction mode capability.
     */
    protected static class TransactionModeCapability<T extends TransactionRequirements> {

        /**
         * The transaction mode.
         */
        private final TransactionMode mode;

        /**
         * The requirements.
         */
        private final T requirements;

        /**
         * Constructs a {@link TransactionModeCapability}.
         *
         * @param mode         the transaction mode
         * @param requirements the requirements
         */
        public TransactionModeCapability(TransactionMode mode, T requirements) {
            this.mode = mode;
            this.requirements = requirements;
        }

        /**
         * Returns the transaction mode.
         *
         * @return the transaction mode
         */
        public TransactionMode getMode() {
            return mode;
        }

        /**
         * Returns the requirements for the transaction mode.
         *
         * @return the requirements
         */
        public T getRequirements() {
            return requirements;
        }
    }
}