/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.service;

/**
 * {@link RefundRequirements} builder.
 *
 * @author Tim Anderson
 */
public class RefundRequirementsBuilder extends TransactionRequirementsBuilder<RefundRequirementsBuilder> {

    /**
     * Determines if the payment processor supports unmatched refunds.
     */
    private boolean supportsUnmatchedRefunds;

    /**
     * Determines if unmatched refunds are supported.
     * <p/>
     * These are refunds without a corresponding payment.
     *
     * @param supportsUnmatchedRefunds if {@code true}, the service supports unmatched refunds,
     *                                 otherwise the refunds must be linked to a payment
     */
    public RefundRequirementsBuilder supportsUnmatchedRefunds(boolean supportsUnmatchedRefunds) {
        this.supportsUnmatchedRefunds = supportsUnmatchedRefunds;
        return this;
    }

    /**
     * Builds the requirements.
     *
     * @return the requirements
     */
    public RefundRequirements build() {
        return new DefaultRefundRequirements(this);
    }

    /**
     * Default {@link RefundRequirements} implementation.
     */
    private static class DefaultRefundRequirements extends DefaultTransactionRequirements
            implements RefundRequirements {

        /**
         * Determines if unmatched refunds are supported.
         */
        private final boolean supportsUnmatchedRefunds;

        /**
         * Constructs a {@link DefaultRefundRequirements}.
         *
         * @param builder the builder to populate this from
         */
        protected DefaultRefundRequirements(RefundRequirementsBuilder builder) {
            super(builder);
            this.supportsUnmatchedRefunds = builder.supportsUnmatchedRefunds;
        }

        /**
         * Determines if unmatched refunds are supported.
         * <p/>
         * These are refunds without a corresponding payment.
         *
         * @return {@code true} if the service supports unmatched refunds, {@code false} if refunds must be linked to
         * a payment
         */
        @Override
        public boolean supportsUnmatchedRefunds() {
            return supportsUnmatchedRefunds;
        }
    }
}