/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.service;

import org.openvpms.paymentprocessor.processor.TransactionMode;

import java.util.Collections;
import java.util.Map;

/**
 * {@link RefundCapabilities} builder.
 *
 * @author Tim Anderson
 */
public class RefundCapabilitiesBuilder
        extends TransactionCapabilitiesBuilder<RefundRequirements, RefundCapabilitiesBuilder> {

    /**
     * Determines if refunds are supported.
     */
    private boolean supported = true;

    /**
     * Builds a {@link RefundCapabilities} indicating that refunds aren't supported.
     *
     * @return the refund capabilities
     */
    public RefundCapabilities unsupported() {
        supported = false;
        return build();
    }

    /**
     * Builds the refund capabilities.
     *
     * @return the refund capabilities
     */
    public RefundCapabilities build() {
        return new DefaultRefundCapabilities(supported, getCapabilities());
    }

    private static class DefaultRefundCapabilities extends DefaultTransactionCapabilities<RefundRequirements>
            implements RefundCapabilities {

        private final boolean supported;

        public DefaultRefundCapabilities(boolean supported, Map<TransactionMode,
                TransactionModeCapability<RefundRequirements>> capabilities) {
            super(supported ? capabilities : Collections.emptyMap());
            this.supported = supported;
        }

        /**
         * Determines if refunds are supported by the payment processor.
         *
         * @return {@code true} if refunds are supported, otherwise {@code false}
         */
        @Override
        public boolean isSupported() {
            return supported;
        }
    }
}