/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.paymentprocessor.service;

import org.openvpms.paymentprocessor.exception.PaymentProcessorException;
import org.openvpms.paymentprocessor.transaction.Payment;
import org.openvpms.paymentprocessor.transaction.Refund;
import org.openvpms.paymentprocessor.transaction.Transaction.Status;

/**
 * Payment processor service.
 *
 * @author Tim Anderson
 */
public interface PaymentProcessorService {

    /**
     * Returns a display name for this service.
     *
     * @return a display name for this service
     * @throws PaymentProcessorException for any error
     */
    String getName();

    /**
     * Returns the payment processor archetype that this service supports.
     *
     * @return an <em>entity.paymentProcessor*</em> archetype
     * @throws PaymentProcessorException for any error
     */
    String getArchetype();

    /**
     * Returns the payment capabilities.
     *
     * @return the payment capabilities
     */
    PaymentCapabilities getPaymentCapabilities();

    /**
     * Returns the refund capabilities.
     *
     * @return the refund capabilities
     */
    RefundCapabilities getRefundCapabilities();

    /**
     * Prepare for submission of a payment.
     * <p/>
     * This ensures that all the details required to successfully submit the payment are present.
     * <p/>
     * Implementations may use this to create or update account details for the customer.
     * <p/>
     * For new payments, or payments that have never been submitted, the status will be {@link Status#PENDING}.
     * For payments that failed submission, the status will be {@link Status#IN_PROGRESS}.
     *
     * @param payment the payment
     * @return the validation status
     * @throws PaymentProcessorException for any error
     */
    ValidationStatus prepare(Payment payment);

    /**
     * Submit a payment.
     * <p/>
     * On successful submission, the status should be set to:
     * <ul>
     *     <li>{@link Status#SUBMITTED SUBMITTED} - if payment will occur some time in the future; or</li>
     *     <li>{@link Status#COMPLETED COMPLETED} - if payment is complete</li>
     * </ul>
     * <p/>
     * If submission fails, and the problem is:
     * <ul>
     *     <li>transient - the status should be left unchanged and an exception thrown. The user can resubmit.</li>
     *     <li>unrecoverable, the status should be set to {@link Status#ERROR ERROR}. Payments with
     *         {@link Status#ERROR ERROR} status will never be resubmitted.</li>
     * </ul>
     * The <em>isNew</em> argument is used to indicate if this is the initial submission, or a resubmission after a
     * failure. Implementations should ensure that when {@code isNew == false}, a duplicate transaction is not
     * submitted.
     *
     * @param payment the payment
     * @param isNew   if {@code true}, this is the first attempt at submitting the payment, otherwise the payment has
     *                been submitted previously, but the submission failed
     * @throws PaymentProcessorException for any error
     */
    void submit(Payment payment, boolean isNew);

    /**
     * Checks the status of a payment.
     * <p/>
     * This is used to determine if an {@link Status#SUBMITTED SUBMITTED} payment has been updated.
     *
     * @param payment the payment
     * @return {@code true} if the payment was updated, otherwise {@code false}
     * @throws PaymentProcessorException for any error
     */
    boolean check(Payment payment);

    /**
     * Cancel a payment.
     *
     * @param payment the payment to cancel
     * @throws PaymentProcessorException for any error
     */
    void cancel(Payment payment);

    /**
     * Prepare for submission of a refund.
     * <p/>
     * This ensures that all the details required to successfully submit the refund are present.
     * <p/>
     * Implementations may use this to create or update account details for the customer.
     * <p/>
     * For new refunds, or refunds that have never been submitted, the status will be {@link Status#PENDING}.
     * For refunds that failed submission, the status will be {@link Status#IN_PROGRESS}.
     *
     * @param refund the refund
     * @return the validation status
     * @throws PaymentProcessorException for any error
     */
    ValidationStatus prepare(Refund refund);

    /**
     * Submit a refund.
     * <p/>
     * On successful submission, the status should be set to:
     * <ul>
     *     <li>{@link Status#SUBMITTED SUBMITTED} - if the refund will occur some time in the future; or</li>
     *     <li>{@link Status#COMPLETED COMPLETED} - if the refund is complete</li>
     * </ul>
     * If submission fails, and the problem is:
     * <ul>
     *     <li>transient - the status should be left unchanged and an exception thrown. The user can resubmit.</li>
     *     <li>unrecoverable, the status should be set to {@link Status#ERROR ERROR}. Payments with
     *         {@link Status#ERROR ERROR} status will never be resubmitted.</li>
     * </ul>
     * The <em>isNew</em> argument is used to indicate if this is the initial submission, or a resubmission after a
     * failure. Implementations should ensure that when {@code isNew == false}, a duplicate transaction is not
     * submitted.
     *
     * @param refund the refund
     * @param isNew  if {@code true}, this is the first attempt at preparing the payment, otherwise the payment has
     *               been submitted previously, but failed
     * @throws PaymentProcessorException for any error
     */
    void submit(Refund refund, boolean isNew);

    /**
     * Checks the status of a refund.
     * <p/>
     * This is used to determine if an {@link Status#IN_PROGRESS} refund has been updated.
     *
     * @param refund the refund
     * @return {@code true} if the refund was updated, otherwise {@code false}
     * @throws PaymentProcessorException for any error
     */
    boolean check(Refund refund);

    /**
     * Cancel a refund.
     *
     * @param refund the refund to cancel
     * @throws PaymentProcessorException for any error
     */
    void cancel(Refund refund);

}