/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2020 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.macro.impl;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.macro.MacroException;

/**
 * A {@code Macro} that evaluates an JXPath expression.
 *
 * @author Tim Anderson
 */
public abstract class AbstractExpressionMacro extends Macro {

    /**
     * The JXPath expression
     */
    private final String expression;


    /**
     * Constructs an {@link AbstractExpressionMacro}.
     *
     * @param lookup  the macro definition
     * @param service the archetype service
     * @throws MacroException if the expression is invalid
     */
    public AbstractExpressionMacro(Lookup lookup, IArchetypeService service) {
        this(service.getBean(lookup));
    }

    /**
     * Constructs an {@link AbstractExpressionMacro}.
     *
     * @param code       the macro code
     * @param name       the macro name
     * @param expression the expression to evaluate
     * @throws MacroException if the expression is invalid
     */
    public AbstractExpressionMacro(String code, String name, String expression) {
        super(code, name);
        if (StringUtils.isEmpty(expression)) {
            throw new MacroException("Expression must be supplied for macro=" + code);
        }
        this.expression = expression;
    }

    /**
     * Constructs an {@link AbstractExpressionMacro}.
     *
     * @param bean the macro definition
     * @throws MacroException if the expression is invalid
     */
    protected AbstractExpressionMacro(IMObjectBean bean) {
        super(bean.getObject(Lookup.class));
        expression = bean.getString("expression");
        if (StringUtils.isEmpty(expression)) {
            throw new MacroException("Expression must be supplied for macro=" + getCode());
        }
    }

    /**
     * Returns the JXPath expression.
     *
     * @return the expression
     */
    public String getExpression() {
        return expression;
    }
}
