/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.insurance.internal.claim;

import org.openvpms.archetype.rules.insurance.InsuranceRules;
import org.openvpms.archetype.rules.patient.MedicalRecordRules;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.insurance.claim.History;
import org.openvpms.insurance.claim.Note;

import java.util.Date;

/**
 * Default implementation of {@link History}.
 *
 * @author Tim Anderson
 */
public class HistoryImpl implements History {

    /**
     * The patient.
     */
    private final Party patient;

    /**
     * The insurer.
     */
    private final Party insurer;

    /**
     * The claim.
     */
    private final Act claim;

    /**
     * The claim practice location.
     */
    private final Reference location;

    /**
     * The minimum treatment date, used to determine which visits to include in the history.
     */
    private final Date minTreatmentDate;

    /**
     * The maximum treatment date, used to determine which visits to include in the history.
     */
    private final Date maxTreatmentDate;

    /**
     * The medical record rules.
     */
    private final MedicalRecordRules recordRules;

    /**
     * The adjusted minimum treatment date, determined by the visit falling on the minTreatmentDate
     */
    private Date fromDate;

    /**
     * The insurance rules.
     */
    private final InsuranceRules insuranceRules;

    /**
     * The adjusted maximum treatment date.
     */
    private Date toDate;

    /**
     * Determines if the to-date is inclusive or exclusive.
     */
    private boolean inclusive;

    /**
     * The query.
     */
    private final NotesQuery query;

    /**
     * The maximum no. of notes to retrieve at a time.
     */
    private static final int PAGE_SIZE = 100;

    /**
     * Constructs a {@link HistoryImpl}.
     *
     * @param patient          the patient
     * @param insurer          the insurer
     * @param claim            the claim
     * @param minTreatmentDate the minimum treatment date, used to determine which visits to include in the history
     * @param maxTreatmentDate the maximum treatment date, used to determine which visits to include in the history
     * @param insuranceRules   the insurance rules
     * @param recordRules      the medical record rules
     * @param service          the archetype service
     */
    public HistoryImpl(Party patient, Party insurer, Act claim, Date minTreatmentDate, Date maxTreatmentDate,
                       InsuranceRules insuranceRules, MedicalRecordRules recordRules, ArchetypeService service) {
        this.patient = patient;
        this.insurer = insurer;
        this.claim = claim;
        location = service.getBean(claim).getTargetRef("location");
        this.minTreatmentDate = minTreatmentDate;
        this.maxTreatmentDate = maxTreatmentDate;
        this.insuranceRules = insuranceRules;
        this.recordRules = recordRules;
        this.query = new NotesQuery(service);
    }

    /**
     * Returns the clinical notes for the patient.
     * <p/>
     * This includes all notes from visits up to the maximum treatment date for the claim conditions.
     *
     * @return the clinical notes
     */
    @Override
    public Iterable<Note> getNotes() {
        return query.query(patient, null, false, getTreatedTo(), isTreatedToInclusive(), PAGE_SIZE);
    }

    /**
     * Returns the clinical notes since the claim prior to the conditions being claimed.
     * <p/>
     * This applies to insurers that have a common insurance service. It returns notes that have not been submitted
     * to the insurance service before, or were part of a cancelled claim. The objective is to reduce redundant
     * history submissions.
     * <p/>
     * If there has been no prior claim, or the insurer isn't linked to an insurance service, all notes up to the time
     * of the claim will be returned.
     *
     * @return the clinical notes
     */
    @Override
    public Iterable<Note> getNotesSinceLastClaim() {
        Date prior = getPriorTreatmentDate();
        boolean inclusiveFrom = false;
        if (prior != null && fromDate != null) {
            inclusiveFrom = DateRules.compareTo(prior, fromDate) == 0;
        }
        return query.query(patient, prior, inclusiveFrom, getTreatedTo(), isTreatedToInclusive(), PAGE_SIZE);
    }

    /**
     * Finds the most recent treatment end date for a patient claim submitted to the insurance service of an insurer,
     * prior to the specified treatment date.
     * <p/>
     * This can be used to avoid sending redundant patient history to an insurance service, if they have already been
     * sent history in a prior claim.
     *
     * @return the prior treatment date, or {@code null} if there is none
     */
    public Date getPriorTreatmentDate() {
        return insuranceRules.getPriorTreatmentDate(patient, insurer, claim, getTreatedFrom());
    }

    /**
     * Returns the minimum treatment date/time, adjusted to the start time of the visit falling on the date, if any.
     * <p/>
     * This is required for situations where users manually enter treatment dates, but they don't correspond
     * to visit start times.
     *
     * @return the adjusted minimum treatment date
     */
    public Date getTreatedFrom() {
        initTreatedDates();
        return fromDate;
    }

    /**
     * Returns the maximum treatment date/time, adjusted to end time of the visit falling on the date, if any.
     * <p/>
     * This is required for situations where users manually enter treatment dates, but they don't correspond
     * to visit end times.
     *
     * @return the adjusted maximum treatment date
     */
    public Date getTreatedTo() {
        initTreatedDates();
        return toDate;
    }

    /**
     * Determines if the {@link #getTreatedTo() treated-to date} is inclusive of visits starting on the date, or
     * exclusive of them.
     *
     * @return {@code true} is inclusive, otherwise {@code false} if it is exclusive
     */
    public boolean isTreatedToInclusive() {
        initTreatedDates();
        return inclusive;
    }

    /**
     * Initialises the treated from and to dates, if they haven't already been determined.
     */
    private void initTreatedDates() {
        if (fromDate == null) {
            // get the visit associated with the minimum treatment date
            Act visit1 = recordRules.getEvent(patient.getObjectReference(), minTreatmentDate, location);
            if (visit1 != null) {
                fromDate = visit1.getActivityStartTime();
                Date to = visit1.getActivityEndTime();
                if (maxTreatmentDate == null) {
                    toDate = to;
                    inclusive = true;
                } else if (to != null) {
                    if (DateRules.compareTo(maxTreatmentDate, to, true) <= 0) {
                        toDate = to;
                        inclusive = true;
                    } else{
                        // maxTreatmentDate > to
                        // get the visit associated with the maximum treatment date
                        Act visit2 = recordRules.getEvent(patient.getObjectReference(), maxTreatmentDate, location);
                        if (visit2 != null && visit2.getActivityEndTime() != null) {
                            toDate = visit2.getActivityEndTime();
                            inclusive = visit2.getActivityStartTime() != null
                                        && visit2.getActivityStartTime().equals(toDate);
                        } else {
                            toDate = maxTreatmentDate;
                        }
                    }
                } else {
                    toDate = maxTreatmentDate;
                }
            } else {
                fromDate = minTreatmentDate;
                toDate = maxTreatmentDate;
            }
            if (!inclusive && fromDate != null && toDate != null) {
                inclusive = DateRules.compareTo(fromDate, toDate, true) == 0;
            }
        }
    }
}