/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.insurance.claim;

import org.openvpms.component.model.lookup.Lookup;

import java.math.BigDecimal;
import java.time.OffsetDateTime;
import java.util.List;

/**
 * Represents a condition being claimed in an insurance claim.
 *
 * @author Tim Anderson
 */
public interface Condition {

    enum Status {
        RESOLVED,     // condition is resolved
        UNRESOLVED,   // condition is ongoing
        DIED,         // animal died from the condition
        EUTHANASED    // animal was euthanased because of the condition
    }

    /**
     * The date when treatment for the condition was started.
     *
     * @return date when treatment for the condition was started
     */
    OffsetDateTime getTreatedFrom();

    /**
     * The date when treatment for the condition was ended.
     *
     * @return date when treatment for the condition was ended
     */
    OffsetDateTime getTreatedTo();

    /**
     * Returns the diagnosis.
     * <p>
     * If no diagnosis is provided, a {@link #getDescription() description} is required.
     *
     * @return the diagnosis. May be {@code null}
     */
    Lookup getDiagnosis();

    /**
     * Returns the condition description.
     * <p>
     * This can provide a short summary of the condition.
     *
     * @return the condition description. May be {@code null}
     */
    String getDescription();

    /**
     * Returns the status of the animal as a result of this condition.
     *
     * @return the status of the animal
     */
    Status getStatus();

    /**
     * Returns the reason for euthanasing the animal, if {@link #getStatus()} is {@code EUTHANASED}.
     *
     * @return the reason for euthanasing the animal
     */
    String getEuthanasiaReason();

    /**
     * Returns the consultation notes.
     *
     * @return the consultation notes
     */
    List<Note> getConsultationNotes();

    /**
     * Returns the discount amount, including tax.
     *
     * @return the discount amount
     */
    BigDecimal getDiscount();

    /**
     * Returns the discount tax amount.
     *
     * @return the discount tax amount
     */
    BigDecimal getDiscountTax();

    /**
     * Returns the total amount, including tax.
     *
     * @return the total amount
     */
    BigDecimal getTotal();

    /**
     * Returns the total tax amount.
     *
     * @return the tax amount
     */
    BigDecimal getTotalTax();

    /**
     * Returns the total amount that has been paid toward invoices being claimed for the condition, excluding amounts
     * that have been allocated by the payment of other claims.
     * <p/>
     * Specifically, this is the sum of allocated amounts of the invoices being claimed, minus any allocations which
     * can be attributed to items in the invoices that have been claimed by other claims.<br/>
     * The other claims must be pre-paid claims or gap claims with status of PAID or NOTIFIED, as this indicates
     * that payments have been allocated.
     *
     * @return the current paid amount
     */
    BigDecimal getCurrentPaid();

    /**
     * Returns the outstanding balance for invoices being claimed for the condition.
     * <p/>
     * This is the condition total - the current paid amount.
     *
     * @return the total amount outstanding
     */
    BigDecimal getCurrentBalance();

    /**
     * Returns the invoices being claimed.
     *
     * @return the invoices being claimed
     */
    List<Invoice> getInvoices();

}
