/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.hl7.impl;

import ca.uhn.hl7v2.model.Message;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.product.Product;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.component.service.lookup.LookupService;
import org.openvpms.hl7.io.Connector;
import org.openvpms.hl7.io.MessageDispatcher;
import org.openvpms.hl7.patient.PatientContext;
import org.openvpms.hl7.pharmacy.Pharmacies;
import org.openvpms.hl7.pharmacy.PharmacyOrderService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.util.Date;

/**
 * Default implementation of the {@link PharmacyOrderService}.
 *
 * @author Tim Anderson
 */
public class PharmacyOrderServiceImpl implements PharmacyOrderService {

    /**
     * The services.
     */
    private final Pharmacies pharmacies;

    /**
     * The message dispatcher.
     */
    private final MessageDispatcher dispatcher;

    /**
     * The message factory.
     */
    private final RDEMessageFactory factory;

    /**
     * The logger.
     */
    private static final Logger log = LoggerFactory.getLogger(PharmacyOrderServiceImpl.class);

    /**
     * Constructs a {@link PharmacyOrderServiceImpl}.
     *
     * @param service    the archetype service
     * @param lookups    the lookup service
     * @param pharmacies the services
     * @param dispatcher the message dispatcher
     */
    public PharmacyOrderServiceImpl(ArchetypeService service, LookupService lookups, Pharmacies pharmacies,
                                    MessageDispatcherImpl dispatcher) {
        this.pharmacies = pharmacies;
        this.dispatcher = dispatcher;
        factory = new RDEMessageFactory(dispatcher.getMessageContext(), service, lookups);
    }

    /**
     * Creates an order, placing it with the specified pharmacy.
     *
     * @param context           the patient context
     * @param product           the product to order
     * @param quantity          the quantity to order
     * @param placerOrderNumber the placer order number, to uniquely identify the order
     * @param date              the order date
     * @param pharmacy          the pharmacy. An <em>entity.HL7ServicePharmacy</em>
     * @return {@code true} if the order was placed
     */
    @Override
    public boolean createOrder(PatientContext context, Product product, BigDecimal quantity, long placerOrderNumber,
                               Date date, Entity pharmacy) {
        boolean result = false;
        Connector connector = getConnector(pharmacy);
        if (connector != null) {
            HL7Mapping config = connector.getMapping();
            Message message = factory.createOrder(context, product, quantity, placerOrderNumber,
                                                  connector.getSendingApplication(), date, config);
            dispatcher.queue(message, connector, config);
            result = true;
        } else if (log.isDebugEnabled()) {
            log.debug("Cannot create order={} for patient={}, product={}, quantity={}, pharmacy={}: no connector",
                      placerOrderNumber, toString(context.getPatient()), toString(product), quantity,
                      toString(pharmacy));
        }
        return result;
    }

    /**
     * Updates an order.
     *
     * @param context           the patient context
     * @param product           the product to order
     * @param quantity          the quantity to order
     * @param placerOrderNumber the placer order number, to uniquely identify the order
     * @param date              the order date
     * @param pharmacy          the pharmacy. An <em>entity.HL7ServicePharmacy</em>
     * @return {@code true} if the order was updated, otherwise {@code false}
     */
    @Override
    public boolean updateOrder(PatientContext context, Product product, BigDecimal quantity, long placerOrderNumber,
                               Date date, Entity pharmacy) {
        boolean result = false;
        Connector connector = getConnector(pharmacy);
        if (connector != null) {
            HL7Mapping config = connector.getMapping();
            Message message = factory.updateOrder(context, product, quantity, placerOrderNumber,
                                                  connector.getSendingApplication(), date, config);
            dispatcher.queue(message, connector, config);
            result = true;
        } else if (log.isDebugEnabled()) {
            log.debug("Cannot update order={} for patient={}, product={}, quantity={}, pharmacy={}: no connector",
                      placerOrderNumber, toString(context.getPatient()), toString(product), quantity,
                      toString(pharmacy));
        }
        return result;
    }

    /**
     * Cancels an order.
     *
     * @param context           the patient context
     * @param product           the product to order
     * @param quantity          the quantity to order
     * @param placerOrderNumber the placer order number, to uniquely identify the order
     * @param date              the order date
     * @param pharmacy          the pharmacy. An <em>entity.HL7ServicePharmacy</em>
     */
    @Override
    public void cancelOrder(PatientContext context, Product product, BigDecimal quantity, long placerOrderNumber,
                            Date date, Entity pharmacy) {
        Connector connector = getConnector(pharmacy);
        if (connector != null) {
            HL7Mapping config = connector.getMapping();
            Message message = factory.cancelOrder(context, product, quantity, placerOrderNumber,
                                                  connector.getSendingApplication(), config, date);
            dispatcher.queue(message, connector, config);
        } else if (log.isDebugEnabled()) {
            log.debug("Cannot cancel order={} for patient={}, product={}, quantity={}, pharmacy={}: no connector",
                      placerOrderNumber, toString(context.getPatient()), toString(product), quantity,
                      toString(pharmacy));
        }
    }

    /**
     * Discontinues an order.
     *
     * @param context           the patient context
     * @param product           the product to order
     * @param quantity          the quantity to order
     * @param placerOrderNumber the placer order number, to uniquely identify the order
     * @param date              the order date
     * @param pharmacy          the pharmacy. An <em>entity.HL7ServicePharmacy</em>
     */
    @Override
    public void discontinueOrder(PatientContext context, Product product, BigDecimal quantity, long placerOrderNumber,
                                 Date date, Entity pharmacy) {
        Connector connector = getConnector(pharmacy);
        if (connector != null) {
            HL7Mapping config = connector.getMapping();
            Message message = factory.discontinueOrder(context, product, quantity, placerOrderNumber,
                                                       connector.getSendingApplication(), config, date);
            dispatcher.queue(message, connector, config);
        } else if (log.isDebugEnabled()) {
            log.debug("Cannot discontinue order={} for patient={}, product={}, quantity={}, pharmacy={}: no connector",
                      placerOrderNumber, toString(context.getPatient()), toString(product), quantity,
                      toString(pharmacy));
        }
    }

    /**
     * Returns a connection for a pharmacy.
     *
     * @param pharmacy the pharmacy
     * @return the connection, or {@code null} if none is found
     */
    private Connector getConnector(Entity pharmacy) {
        return pharmacies.getSender(pharmacy);
    }

    /**
     * Helper to return a string representation of an {@link IMObject} for debugging purposes.
     *
     * @param object the object. May be {@code null}
     * @return the string
     */
    private String toString(IMObject object) {
        return (object != null) ? object.getName() + " (" + object.getId() + ")" : "null";
    }
}
