/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2022 (C) OpenVPMS Ltd. All Rights Reserved.
 */
package org.openvpms.component.business.service.lookup;

import org.apache.commons.resources.Messages;
import org.openvpms.component.exception.OpenVPMSException;


/**
 * Exceptions raised by the {@link ILookupService}.
 *
 * @author <a href="mailto:support@openvpms.org">OpenVPMS Team</a>
 */
public class LookupServiceException extends OpenVPMSException {

    /**
     * An enumeration of error codes.
     */
    public enum ErrorCode {
        LookupServiceNotSet,
        CannotReplaceArchetypeMismatch
    }

    /**
     * The error code.
     */
    private final ErrorCode errorCode;

    /**
     * Serial version identifier.
     */
    private static final long serialVersionUID = 1L;

    /**
     * The error messages.
     */
    private static final Messages messages
            = Messages.getMessages("org.openvpms.component.business.service.lookup.errmessages");

    /**
     * Constructs a {@link LookupServiceException}.
     *
     * @param errorCode the error code
     */
    public LookupServiceException(ErrorCode errorCode) {
        super(org.openvpms.component.i18n.Messages.create(messages.getMessage(errorCode.toString())));
        this.errorCode = errorCode;
    }

    /**
     * Constructs a {@link LookupServiceException}.
     *
     * @param errorCode the error code
     * @param params    the parameters used to render the message associated with the error code
     */
    public LookupServiceException(ErrorCode errorCode, Object... params) {
        super(org.openvpms.component.i18n.Messages.create(messages.getMessage(errorCode.toString(), params)));
        this.errorCode = errorCode;
    }

    /**
     * Constructs a {@link LookupServiceException}.
     *
     * @param errorCode the error code
     * @param cause     the root exception
     */
    public LookupServiceException(ErrorCode errorCode, Throwable cause) {
        super(org.openvpms.component.i18n.Messages.create(messages.getMessage(errorCode.toString())), cause);
        this.errorCode = errorCode;
    }

    /**
     * Constructs a {@code LookupServiceException}.
     *
     * @param errorCode the error code
     * @param cause     the root exception
     * @param params    additional information required to render the message
     */
    public LookupServiceException(ErrorCode errorCode, Throwable cause, Object... params) {
        super(org.openvpms.component.i18n.Messages.create(messages.getMessage(errorCode.toString(), params)), cause);
        this.errorCode = errorCode;
    }

    /**
     * Returns the error code.
     *
     * @return the error code
     */
    public ErrorCode getErrorCode() {
        return errorCode;
    }

}
