/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.component.business.service.archetype.helper;

import org.openvpms.component.business.domain.im.archetype.descriptor.ArchetypeDescriptor;
import org.openvpms.component.business.domain.im.archetype.descriptor.NodeDescriptor;
import org.openvpms.component.business.service.archetype.MapIMObjectGraph;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Helper to copy {@link IMObject} instances.
 *
 * @author Tim Anderson
 */
public class IMObjectCopier {

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The copy handler.
     */
    private final IMObjectCopyHandler handler;

    /**
     * Map of original -> copied objects, to avoid duplicate copying.
     */
    private Map<Reference, Copy> copies;


    /**
     * Constructs a {@link IMObjectCopier}.
     *
     * @param handler the copy handler
     * @param service the archetype service
     */
    public IMObjectCopier(IMObjectCopyHandler handler, ArchetypeService service) {
        this.handler = handler;
        this.service = service;
    }

    /**
     * Copy an object, returning a list containing the copy, and any copied
     * child references.
     * <p>
     * Any derived values will be populated on the returned objects.
     *
     * @param object the object to copy
     * @return a copy of {@code object}, and any copied child references.
     * The copy of {@code object} is the first element in the returned
     * list
     */
    public List<IMObject> apply(IMObject object) {
        List<IMObject> result = new ArrayList<>();
        copies = new HashMap<>();
        IMObject target = apply(object, result, false);
        result.add(0, target);
        return result;
    }

    /**
     * Copies an object, returning an {@link IMObjectGraph} containing the copy and its related objects.
     * <p>
     * Any derived values will be populated on the returned objects.
     *
     * @param object the object to copy
     * @return the copied objects
     */
    public IMObjectGraph copy(IMObject object) {
        List<IMObject> result = apply(object);
        IMObject primary = result.get(0);
        return new MapIMObjectGraph(primary, result);
    }

    /**
     * Returns a builder to create an {@link IMObjectCopier}.
     *
     * @param service the archetype service
     * @return a new builder
     */
    public static IMObjectCopierBuilder newCopier(ArchetypeService service) {
        return new IMObjectCopierBuilder(service);
    }

    /**
     * Apply the copier to an object, copying it or returning it unchanged, as
     * determined by the {@link IMObjectCopyHandler}.
     *
     * @param source   the source object
     * @param children a list of child objects created during copying
     * @param save     determines if child objects should be saved
     * @return a copy of {@code source} if the handler indicates it should
     * be copied; otherwise returns {@code source} unchanged
     */
    protected IMObject apply(IMObject source, List<IMObject> children, boolean save) {
        IMObject target = handler.getObject(source, service);
        if (target != null) {
            // cache the references to avoid copying the same object twice
            Copy copy = new Copy(target);
            copies.put(source.getObjectReference(), copy);

            if (target != source) {
                doCopy(source, target, children, save);
            }
            copy.complete();
        }
        return target;
    }

    /**
     * Performs a copy of an object.
     *
     * @param source   the object to copy
     * @param target   the target to copy to
     * @param children a list of child objects created during copying
     * @param save     determines if child objects should be saved
     */
    protected void doCopy(IMObject source, IMObject target, List<IMObject> children, boolean save) {
        ArchetypeDescriptor sourceType = DescriptorHelper.getArchetypeDescriptor(source, service);
        ArchetypeDescriptor targetType = DescriptorHelper.getArchetypeDescriptor(target, service);

        // copy the nodes
        for (NodeDescriptor sourceDesc : sourceType.getAllNodeDescriptors()) {
            NodeDescriptor targetDesc = handler.getNode(sourceType, sourceDesc, targetType);
            if (targetDesc != null) {
                if (sourceDesc.isObjectReference()) {
                    Reference ref = (Reference) sourceDesc.getValue(source);
                    if (ref != null) {
                        ref = copyReference(ref, source, children, save);
                        sourceDesc.setValue(target, ref);
                    }
                } else if (!sourceDesc.isCollection()) {
                    targetDesc.setValue(target, sourceDesc.getValue(source));
                } else {
                    copyChildren(source, target, children, save, sourceDesc, targetDesc);
                }
            }
        }
        // derive any values in the target
        service.deriveValues(target);
    }

    /**
     * Handle a missing reference.
     * <p/>
     * This implementation throws {@link IMObjectCopierException}. Subclasses can throw an exception, or ignore it.
     *
     * @param reference the reference
     * @param parent    the parent
     */
    protected void missingReference(Reference reference, IMObject parent) {
        throw new IMObjectCopierException(IMObjectCopierException.ErrorCode.ObjectNotFound, reference,
                                          parent.getObjectReference());
    }

    /**
     * Copies collections.
     *
     * @param source     the object to copy
     * @param target     the target to copy to
     * @param children   a list of child objects created during copying
     * @param save       determines if child objects should be saved
     * @param sourceDesc the source collection node descriptor
     * @param targetDesc the target collection node descriptor
     */
    private void copyChildren(IMObject source, IMObject target, List<IMObject> children, boolean save,
                              NodeDescriptor sourceDesc, NodeDescriptor targetDesc) {
        for (IMObject child : sourceDesc.getChildren(source)) {
            IMObject value;
            Copy copy = null;
            if (sourceDesc.isParentChild()) {
                copy = copies.get(child.getObjectReference());
                if (copy == null) {
                    value = apply(child, children, save);
                } else {
                    // referencing an object already present in another collection
                    value = copy.getObject();
                }
            } else {
                value = child;
            }
            if (value != null) {
                if (copy == null || copy.isComplete()) {
                    targetDesc.addChildToCollection((org.openvpms.component.business.domain.im.common.IMObject) target,
                                                    value);
                } else {
                    // can't safely add an incomplete object to the collection, so queue it for later
                    copy.queue(target, targetDesc);
                }
            }
        }
    }

    /**
     * Helper to copy the object referred to by a reference, and return the new reference.
     *
     * @param reference the reference
     * @param parent    the parent object
     * @param children  a list of child objects created during copying
     * @param save      determines if child objects should be saved
     * @return a new reference, or one from {@code references} if the reference has already been copied
     */
    private Reference copyReference(Reference reference, IMObject parent, List<IMObject> children, boolean save) {
        Copy copy = copies.get(reference);
        IMObject object = null;
        if (copy != null) {
            object = copy.getObject();
        } else {
            IMObject original = service.get(reference);
            if (original == null) {
                missingReference(reference, parent);
            } else {
                object = apply(original, children, save);
                if (object != original && object != null) {
                    // child was copied
                    children.add(object);
                    if (save) {
                        service.save(object);
                    }
                }
            }
        }
        return (object != null) ? object.getObjectReference() : null;
    }

    /**
     * Manages the state of a copied object.
     */
    private static class Copy {

        /**
         * The copied object.
         */
        private final IMObject object;

        /**
         * Determines if the object is complete.
         */
        private boolean complete;

        /**
         * A queue of collection objects.
         */
        private List<CollectionAdder> queue;

        /**
         * Creates a new {@code Copy}.
         *
         * @param object the copied object
         */
        Copy(IMObject object) {
            this.object = object;
        }

        /**
         * Returns the copied object.
         *
         * @return the copied object
         */
        public IMObject getObject() {
            return object;
        }

        /**
         * Determines if the object is complete.
         *
         * @return {@code true} if the object is complete, otherwise {@code false}
         */
        public boolean isComplete() {
            return complete;
        }

        /**
         * Marks the object as being complete, adding it to any queued collections.
         */
        public void complete() {
            this.complete = true;
            if (queue != null) {
                for (CollectionAdder adder : queue) {
                    adder.add();
                }
                queue.clear();
            }
        }

        /**
         * Queues the object for addition to a collection.
         * <p>
         * This should be used to queue incomplete objects for addition until such time as they are complete.
         *
         * @param target     the target object owns the collection
         * @param descriptor the collection node descriptor
         */
        public void queue(IMObject target, NodeDescriptor descriptor) {
            if (queue == null) {
                queue = new ArrayList<>();
            }
            queue.add(new CollectionAdder(target, descriptor, object));
        }
    }

    /**
     * Helper to add an object to a collection.
     */
    private static class CollectionAdder {

        /**
         * The parent object.
         */
        private final IMObject parent;

        /**
         * The collection node descriptor.
         */
        private final NodeDescriptor descriptor;

        /**
         * The object to add.
         */
        private final IMObject value;

        /**
         * Constructs a {@link CollectionAdder}.
         *
         * @param parent     the parent object
         * @param descriptor the collection node descriptor
         * @param value      the value to add
         */
        CollectionAdder(IMObject parent, NodeDescriptor descriptor, IMObject value) {
            this.parent = parent;
            this.descriptor = descriptor;
            this.value = value;
        }

        /**
         * Adds the object to the collection.
         */
        public void add() {
            descriptor.addChildToCollection((org.openvpms.component.business.domain.im.common.IMObject) parent, value);
        }
    }
}