/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */


package org.openvpms.component.business.service.archetype;

import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.i18n.FrameworkMessages;
import org.openvpms.component.service.archetype.ValidationError;

import java.util.Collections;
import java.util.List;

/**
 * This exception is raised when we are validating objects
 *
 * @author Jim Alateras
 */
public class ValidationException extends OpenVPMSException {

    /**
     * The errors.
     */
    private final List<org.openvpms.component.service.archetype.ValidationError> errors;

    /**
     * Generated SUID
     */
    private static final long serialVersionUID = 2L;

    /**
     * Constructs a {@link ValidationException}.
     *
     * @param archetype the main archetype
     * @param error    the validation error
     */
    public ValidationException(String archetype, ValidationError error) {
        this(archetype, Collections.singletonList(error));
    }

    /**
     * Constructs a {@link ValidationException}.
     *
     * @param archetype the main archetype
     * @param errors    the list of validation errors
     */
    public ValidationException(String archetype, List<ValidationError> errors) {
        super(FrameworkMessages.failedToValidateObject(archetype));
        this.errors = errors;
    }

    /**
     * Returns the validation errors.
     *
     * @return the validation errors
     */
    public List<org.openvpms.component.service.archetype.ValidationError> getErrors() {
        return errors;
    }

    /* (non-Javadoc)
     * @see java.lang.Throwable#toString()
     */
    @Override
    public String toString() {
        StringBuilder buf = new StringBuilder(getMessage()).append("\n");
        for (org.openvpms.component.service.archetype.ValidationError error : errors) {
            buf.append("\tnode:")
                    .append(error.getNode())
                    .append(" msg:")
                    .append(error.getMessage())
                    .append("\n");
        }
        return buf.toString();
    }

}
