/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.component.business.domain.im.archetype.descriptor;

import org.apache.commons.beanutils.MethodUtils;
import org.apache.commons.jxpath.JXPathContext;
import org.apache.commons.jxpath.JXPathTypeConversionException;
import org.apache.commons.jxpath.util.TypeConverter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.openvpms.component.business.domain.archetype.ArchetypeId;
import org.openvpms.component.business.domain.archetype.ReadOnlyArchetypeId;
import org.openvpms.component.business.domain.im.common.IMObject;
import org.openvpms.component.business.domain.im.common.IMObjectReference;
import org.openvpms.component.business.domain.im.datatypes.property.AssertionProperty;
import org.openvpms.component.business.domain.im.datatypes.property.PropertyCollection;
import org.openvpms.component.business.domain.im.datatypes.property.PropertyList;
import org.openvpms.component.business.domain.im.datatypes.property.PropertyMap;
import org.openvpms.component.business.domain.im.datatypes.quantity.Money;
import org.openvpms.component.model.archetype.ArchetypeRange;
import org.openvpms.component.model.archetype.NamedProperty;
import org.openvpms.component.model.archetype.Units;
import org.openvpms.component.system.common.jxpath.JXPathHelper;
import org.openvpms.component.system.common.jxpath.OpenVPMSTypeConverter;
import org.openvpms.component.system.common.util.StringUtilities;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;


/**
 * Node descriptor.
 *
 * @author Jim Alateras
 */
public class NodeDescriptor extends Descriptor implements org.openvpms.component.model.archetype.NodeDescriptor {

    /**
     * The default display length if one is not defined in the node definition
     */
    public static final int DEFAULT_DISPLAY_LENGTH = 50;

    /**
     * The name given to the object id node
     */
    public static final String IDENTIFIER_NODE_NAME = "id";

    /**
     * Representation of max cardinality as a string
     */
    public static final String UNBOUNDED_AS_STRING = "*";

    /**
     * Contains a list of {@link AssertionDescriptor} instances
     */
    private Map<String, org.openvpms.component.model.archetype.AssertionDescriptor> assertionDescriptors
            = new LinkedHashMap<>();

    /**
     * This is an option property, which is required for nodes that represent
     * collections. It is the name that denotes the individual elements stored
     * in the collection.
     */
    private String baseName;

    /**
     * Cache the clazz. Do not access this directly. Use the {@link #getClassType()}
     * method instead.
     */
    private transient Class<?> clazz;

    /**
     * The default value.
     */
    private String defaultValue;

    /**
     * This is a jxpath expression, which is used to determine the value of the node.
     */
    private String derivedValue;

    /**
     * This is the display name, which is only supplied if it is different to the node name.
     */
    private String displayName;

    /**
     * The index of this discriptor within the collection
     */
    private int index;

    /**
     * Determine whether the value for this node is derived
     */
    private boolean isDerived = false;

    /**
     * Attribute, which defines whether this node is hidden or can be displayed
     */
    private boolean isHidden = false;

    /**
     * Indicates that the collection type is a parentChild relationship, which
     * is the default for a collection. If this attribute is set to false then
     * the child lifecycle is independent of the parent lifecycle. This
     * attribute is only meaningful for a collection
     */
    private boolean isParentChild = true;

    /**
     * Indicates whether the descriptor is readOnly
     */
    private boolean isReadOnly = false;

    /**
     * Indicates whether the node value represents an array
     */
    private boolean isArray = false;

    /**
     * The maximum cardinality, which defaults to 1
     */
    private int maxCardinality = 1;

    /**
     * The maximum length
     */
    private int maxLength;

    /**
     * The minimum cardinality, which defaults to 0
     */
    private int minCardinality = 0;

    /**
     * The minimum length
     */
    private int minLength;

    /**
     * A node can have other nodeDescriptors to define a nested structure
     */
    private Map<String, NodeDescriptor> nodeDescriptors = new LinkedHashMap<>();

    /**
     * The XPath/JXPath expression that is used to resolve this node within the
     * associated domain object.
     */
    private String path;

    /**
     * The fully qualified class name that defines the node type
     */
    private String type;

    /**
     * The filter is only valid for collections and defines the subset of
     * the collection that this node refers too.  The filter is an archetype
     * shortName, which can also be in the form of a regular expression
     * <p>
     * The modeFilter is a regex compliant filter
     */
    private String filter;

    private String modFilter;

    /**
     * The parent node descriptor. May be {@code null}.
     */
    private NodeDescriptor parent;

    /**
     * The archetype that this descriptor belongs to. May be {@code null}.
     */
    private ArchetypeDescriptor archetype;

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 2L;

    /**
     * Type converter.
     */
    private static final TypeConverter CONVERTER = new OpenVPMSTypeConverter();

    private static final ArchetypeId NODE = new ReadOnlyArchetypeId("descriptor.node.1.0");

    private static final ArchetypeId COLLECTION_NODE = new ReadOnlyArchetypeId("descriptor.collectionNode");

    /**
     * Default constructor.
     */
    public NodeDescriptor() {
        // no-op
    }

    /**
     * Returns the archetype Id. For nodes that have child nodes, returns
     * <em>descriptor.collectionNode.1.0</em>, otherwise returns
     * <em>descriptor.node.1.0</em>.
     *
     * @return the archetype Id.
     */
    @Override
    public ArchetypeId getArchetypeId() {
        return (nodeDescriptors == null || nodeDescriptors.isEmpty()) ? NODE : COLLECTION_NODE;
    }

    /**
     * Adds an assertion descriptor to this node.
     *
     * @param descriptor the assertion descriptor to add
     */
    @Override
    public void addAssertionDescriptor(org.openvpms.component.model.archetype.AssertionDescriptor descriptor) {
        assertionDescriptors.put(descriptor.getName(), descriptor);
    }

    /**
     * Adds a value object to this node descriptor using the specified
     * {@link IMObject} as the context. If this node descriptor is not of type
     * collection, or the context object is null it will raise an exception.
     *
     * @param context the context object, which will be the target of the add
     * @param value   the value element to add
     * @throws DescriptorException if it fails to complete this request
     */
    public void addValue(org.openvpms.component.model.object.IMObject context, Object value) {
        addChildToCollection((IMObject) context, value);
    }

    /**
     * Add a child object to this node descriptor using the specified
     * {@link IMObject} as the context. If this node descriptor is not of type
     * collection, or the context object is null it will raise an exception.
     *
     * @param context the context object, which will be the target of the add
     * @param child   the child element to add
     * @throws DescriptorException if it fails to complete this request
     */
    public void addChildToCollection(IMObject context, Object child) {
        if (context == null) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.FailedToAddChildElement,
                    getName());
        }

        if (!isCollection()) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.FailedToAddChildElement,
                    getName());
        }

        // retrieve the value at that node
        Object obj = JXPathHelper.newContext(context).getValue(getPath());

        try {
            if (StringUtils.isEmpty(baseName)) {
                // no base name specified look at the type to determine what method to call
                Class<?> tClass = getClassType();
                if (Collection.class.isAssignableFrom(tClass)) {
                    MethodUtils.invokeMethod(obj, "add", child);
                } else if (Map.class.isAssignableFrom(tClass)) {
                    MethodUtils.invokeMethod(obj, "put", new Object[]{child, child});
                } else {
                    throw new DescriptorException(
                            DescriptorException.ErrorCode.FailedToAddChildElement,
                            getName());
                }
            } else {
                // if a baseName has been specified then prepend 'add' to the
                // base name and execute the derived method on context object
                String methodName = "add" + StringUtils.capitalize(baseName);

                // TODO This is a tempoaray fix until we resolve the discrepency
                // with collections.
                if (obj instanceof IMObject) {
                    MethodUtils.invokeMethod(obj, methodName, child);
                } else {
                    MethodUtils.invokeMethod(context, methodName, child);
                }
            }
        } catch (Exception exception) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.FailedToAddChildElement,
                    exception, getName());
        }
    }

    /**
     * Add a child node descriptor
     *
     * @param child the child node descriptor to add
     * @throws DescriptorException if the node is a duplicate
     */
    public void addNodeDescriptor(NodeDescriptor child) {
        if (nodeDescriptors.containsKey(child.getName())) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.DuplicateNodeDescriptor,
                    child.getName(), getName());
        }
        nodeDescriptors.put(child.getName(), child);
        child.setParent(this);
    }

    /*
     * (non-Javadoc)
     *
     * @see org.openvpms.component.business.domain.im.archetype.descriptor.Descriptor#clone()
     */
    @Override
    public Object clone() throws CloneNotSupportedException {
        NodeDescriptor copy = (NodeDescriptor) super.clone();
        copy.assertionDescriptors = new LinkedHashMap<>(this.assertionDescriptors);
        copy.nodeDescriptors = new LinkedHashMap<>(this.nodeDescriptors);
        return copy;
    }

    /**
     * Check whether this assertion type is defined for this node
     *
     * @param type the assertion type
     * @return boolean
     */
    @Override
    public boolean containsAssertionType(String type) {
        return assertionDescriptors.containsKey(type);
    }

    /**
     * Derive the node value for the specified {@link NodeDescriptor}. If the
     * node does not support derived value or the value cannot be derived then
     * raise an exception.
     *
     * @param object the {@link IMObject}
     * @throws FailedToDeriveValueException if the node is not declared to support dervied value or
     *                                      the value cannot be derived correctly.
     */
    public void deriveValue(org.openvpms.component.model.object.IMObject object) {
        if (!isDerived()) {
            throw new FailedToDeriveValueException(
                    FailedToDeriveValueException.ErrorCode.DerivedValueUnsupported,
                    new Object[]{getName()});
        }

        // attempt to derive the value
        try {
            JXPathContext context = JXPathHelper.newContext(object);
            Object value = context.getValue(this.getDerivedValue());
            setValue(object, context, value);
        } catch (Exception exception) {
            throw new FailedToDeriveValueException(
                    FailedToDeriveValueException.ErrorCode.FailedToDeriveValue,
                    new Object[]{getName(), getPath(), getDerivedValue()},
                    exception);
        }
    }

    /**
     * Return an array of short names or short name regular expression that are
     * associated with the archetypeRange assertion. If the node does not have
     * such an assertion then return a zero length string array
     *
     * @return String[] the array of short names
     */
    @Override
    public String[] getArchetypeRange() {
        org.openvpms.component.model.archetype.AssertionDescriptor desc = assertionDescriptors.get("archetypeRange");
        if (desc != null) {
            ArrayList<String> range = new ArrayList<>();
            PropertyList archetypes = (PropertyList) desc.getPropertyMap().getProperties().get("archetypes");
            for (org.openvpms.component.model.archetype.NamedProperty property : archetypes.getProperties()) {
                AssertionProperty shortName = (AssertionProperty) ((PropertyMap) property).getProperties().get(
                        "shortName");
                range.add(shortName.getValue());
            }
            return range.toArray(new String[0]);
        } else {
            return new String[0];
        }
    }

    /**
     * Returns the archetypes that this node supports, determined by its <em>archetypeRange</em> assertion if present,
     * or its {@link #getFilter() filter}.
     * <p/>
     * If both are present, the assertion takes precedence.
     *
     * @return the archetypes that this node supports
     */
    @Override
    public ArchetypeRange getArchetypes() {
        ArchetypeRange result;
        org.openvpms.component.model.archetype.AssertionDescriptor desc = assertionDescriptors.get("archetypeRange");
        if (desc != null) {
            result = new ArchetypeRangeAssertion(desc);
        } else if (filter != null) {
            result = new ArchetypeList(filter);
        } else {
            result = ArchetypeList.EMPTY;
        }
        return result;
    }

    /**
     * Retrieve the assertion descriptor with the specified type or null if one
     * does not exist.
     *
     * @param type the type of the assertion descriptor
     * @return AssertionDescriptor
     */
    @Override
    public org.openvpms.component.model.archetype.AssertionDescriptor getAssertionDescriptor(String type) {
        return assertionDescriptors.get(type);
    }

    /**
     * Return the assertion descriptors as a map
     *
     * @return Returns the assertionDescriptors.
     */
    @Override
    public Map<String, org.openvpms.component.model.archetype.AssertionDescriptor> getAssertionDescriptors() {
        return assertionDescriptors;
    }

    /**
     * Return the assertion descriptors in index order.
     *
     * @return the assertion descriptors, ordered on index
     */
    @SuppressWarnings("unchecked")
    public List<AssertionDescriptor> getAssertionDescriptorsInIndexOrder() {
        List<org.openvpms.component.model.archetype.AssertionDescriptor> adescs
                = new ArrayList<>(assertionDescriptors.values());
        adescs.sort(new AssertionDescriptorIndexComparator());
        return (List<AssertionDescriptor>) (List<?>) adescs;
    }

    /**
     * Return the assertion descriptors as an array.
     *
     * @return the assertion descriptors
     */
    public AssertionDescriptor[] getAssertionDescriptorsAsArray() {
        return assertionDescriptors.values().toArray(new AssertionDescriptor[0]);
    }


    /**
     * <br/>
     * NOTE: this is used by castor serialisation
     *
     * @param assertionDescriptors The assertionDescriptors to set.
     */
    public void setAssertionDescriptorsAsArray(
            AssertionDescriptor[] assertionDescriptors) {
        this.assertionDescriptors = new LinkedHashMap<>();
        int i = 0;
        for (AssertionDescriptor descriptor : assertionDescriptors) {
            descriptor.setIndex(i++);
            addAssertionDescriptor(descriptor);
        }
    }

    /**
     * @return Returns the baseName.
     */
    @Override
    public String getBaseName() {
        return baseName;
    }

    /**
     * Return the children {@link IMObject} instances that are part of
     * a collection. If the NodeDescriptor does not denote a collection then
     * a null list is returned.
     * <p>
     * Furthermore if this is a collection and the filter attribute has been
     * specified then return a subset of the children; those matching the
     * filter.
     *
     * @param target the target object.
     * @return List<IMObject>
     * the list of children, an empty list or  null
     */
    @SuppressWarnings("unchecked")
    public List<IMObject> getChildren(org.openvpms.component.model.object.IMObject target) {
        return (List<IMObject>) (List<?>) getValues(target);
    }

    /**
     * Return the {@link IMObject} instances that are part of a collection. If the NodeDescriptor does not denote a
     * collection then a null list is returned.
     * <p>
     * Furthermore if this is a collection and the filter attribute has been
     * specified then return a subset of the children; those matching the
     * filter.
     *
     * @param target the target object.
     * @return the values, or {@code null} if the node isn't a colection
     */
    @SuppressWarnings("unchecked")
    public List<org.openvpms.component.model.object.IMObject>
    getValues(org.openvpms.component.model.object.IMObject target) {
        List<org.openvpms.component.model.object.IMObject> result = null;
        if (isCollection()) {
            try {
                Object obj = JXPathHelper.newContext(target).getValue(getPath());
                if (obj == null) {
                    result = new ArrayList<>();
                } else if (obj instanceof Collection) {
                    result = new ArrayList<>((Collection) obj);
                } else if (obj instanceof PropertyCollection) {
                    result = new ArrayList<>(((PropertyCollection) obj).values());
                } else if (obj instanceof Map) {
                    result = new ArrayList<>(((Map) obj).values());
                }

                // filter the children
                result = filterChildren(result);
            } catch (Exception exception) {
                throw new DescriptorException(
                        DescriptorException.ErrorCode.FailedToGetChildren,
                        exception, target.getName(), getName(), getPath());
            }
        }
        return result;
    }

    /**
     * Returns the class for the specified type.
     *
     * @return the class, or {@code null} if {@link #getType()} returns empty/null
     */
    @Override
    public Class<?> getClassType() {
        return getClazz();
    }

    /**
     * Returns the class for the specified type.
     *
     * @return the class, or {@code null} if {@link #getType()} returns
     * empty/null
     * @throws DescriptorException if the class can't be loaded
     */
    public Class<?> getClazz() {
        if (clazz == null) {
            synchronized (this) {
                clazz = getClass(type);
            }
        }
        return clazz;
    }

    /**
     * @return Returns the defaultValue.
     */
    @Override
    public String getDefaultValue() {
        return defaultValue;
    }

    /**
     * @return Returns the derivedValue.
     */
    @Override
    public String getDerivedValue() {
        return derivedValue;
    }

    /**
     * Return the length of the displayed field. Not currently defined in
     * archetype so set to minimum of maxlength or DEFAULT_DISPLAY_LENGTH. Used
     * for Strings or Numerics.
     *
     * @return int the display length
     */
    public int getDisplayLength() {
        return DEFAULT_DISPLAY_LENGTH;
    }

    /**
     * @return Returns the displayName.
     */
    @Override
    public String getDisplayName() {
        String result = displayName;
        if (StringUtils.isEmpty(result)) {
            String name = getName();
            if (name != null) {
                result = StringUtilities.unCamelCase(name);
            }
        }
        return result;
    }

    /**
     * @return Returns the filter.
     */
    @Override
    public String getFilter() {
        return filter;
    }

    /**
     * @return Returns the index.
     */
    @Override
    public int getIndex() {
        return index;
    }

    /**
     * @return Returns the maxCardinality.
     */
    @Override
    public int getMaxCardinality() {
        return maxCardinality;
    }

    /**
     * The getter that returns the max cardinality as a string
     *
     * @return String
     */
    public String getMaxCardinalityAsString() {
        if (maxCardinality == UNBOUNDED) {
            return UNBOUNDED_AS_STRING;
        } else {
            return Integer.toString(maxCardinality);
        }
    }

    /**
     * @return Returns the maxLength.
     */
    @Override
    public int getMaxLength() {
        return maxLength <= 0 ? DEFAULT_MAX_LENGTH : maxLength;
    }

    /**
     * Return the maximum value of the node. If no maximum defined for node then
     * return 0. Only valid for numeric nodes.
     *
     * @return Number the minimum value
     * @throws DescriptorException a runtim exception
     */
    @Override
    public Number getMaxValue() {
        Number number = null;
        if (isNumeric()) {
            org.openvpms.component.model.archetype.AssertionDescriptor descriptor
                    = assertionDescriptors.get("numericRange");
            if (descriptor != null) {
                number = NumberUtils.createNumber(getValue(descriptor.getPropertyMap().getProperties(), "maxValue"));
            }
        } else {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.UnsupportedOperation,
                    "getMaxValue", getType());
        }

        return number;
    }

    /**
     * @return Returns the minCardinality.
     */
    @Override
    public int getMinCardinality() {
        return minCardinality;
    }

    /**
     * @return Returns the minLength.
     */
    @Override
    public int getMinLength() {
        return minLength;
    }

    /**
     * Return the minimum value of the node. If no minimum defined for node then
     * return 0. Only valid for numeric nodes.
     *
     * @return Number the minimum value
     * @throws DescriptorException a runtim exception
     */
    public Number getMinValue() {
        Number number = null;
        if (isNumeric()) {
            org.openvpms.component.model.archetype.AssertionDescriptor descriptor
                    = assertionDescriptors.get("numericRange");
            if (descriptor != null) {
                number = NumberUtils.createNumber(getValue(descriptor.getPropertyMap().getProperties(), "minValue"));
            }
        } else {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.UnsupportedOperation,
                    "getMinValue", getType());
        }

        return number;
    }

    /**
     * Return the number of children node descriptors
     *
     * @return int
     */
    public int getNodeDescriptorCount() {
        return (nodeDescriptors == null) ? 0 : nodeDescriptors.size();
    }

    /**
     * Return the {@link NodeDescriptor} instances as a map of name and
     * descriptor
     *
     * @return Returns the nodeDescriptors.
     */
    public Map<String, NodeDescriptor> getNodeDescriptors() {
        return nodeDescriptors;
    }

    /**
     * @return Returns the nodeDescriptors.
     */
    public NodeDescriptor[] getNodeDescriptorsAsArray() {
        return nodeDescriptors.values().toArray(new NodeDescriptor[0]);
    }

    /**
     * @return Returns the path.
     */
    @Override
    public String getPath() {
        return path;
    }

    /**
     * Return the regular expression associated with the node. Only valid for
     * string nodes.
     *
     * @return String regular expression pattern
     */
    @Override
    public String getStringPattern() {
        String expression = null;
        if (isString()) {
            org.openvpms.component.model.archetype.AssertionDescriptor descriptor
                    = assertionDescriptors.get("regularExpression");
            if (descriptor != null) {
                expression = getValue(descriptor.getPropertyMap().getProperties(), "expression");
            }
        } else {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.UnsupportedOperation,
                    "getMinValue", getType());
        }

        return expression;
    }

    /**
     * Determines if this node has units.
     *
     * @return the units, or {@code null} if the node has no units
     */
    @Override
    public Units getUnits() {
        Units result = null;
        org.openvpms.component.model.archetype.AssertionDescriptor descriptor
                = assertionDescriptors.get("units");
        if (descriptor != null) {
            Map<String, NamedProperty> properties = descriptor.getPropertyMap().getProperties();
            String node = getValue(properties, "node");
            String displayName = getValue(properties, "displayName");
            if (node != null || displayName != null) {
                result = new UnitsImpl(node, displayName);
            }
        }
        return result;
    }

    /**
     * @return Returns the typeName.
     */
    @Override
    public String getType() {
        return type;
    }

    /**
     * This will return the node value for the supplied {@link IMObject}. If
     * the node is derived then it will return the derived value. If the node is
     * not dervied then it will use the path to return the value.
     *
     * @param context the context object to work from
     * @return Object the returned object
     */
    public Object getValue(org.openvpms.component.model.object.IMObject context) {
        Object value;
        if (isDerived()) {
            String expression = getDerivedValue();
            if (expression != null) {
                value = JXPathHelper.newContext(context).getValue(expression);
            } else {
                value = null;
            }
        } else {
            if (isCollection()) {
                value = getChildren(context);
            } else {
                value = JXPathHelper.newContext(context).getValue(getPath());
            }
        }

        return transform(value);
    }

    /**
     * Returns the value of this node given the specified context.
     *
     * @param context the context to use
     * @return Object
     * the returned object
     */
    public Object getValue(JXPathContext context) {
        Object value = null;
        if (context != null) {
            if (isDerived()) {
                value = context.getValue(getDerivedValue());
            } else {
                if (isCollection()) {
                    value = getChildren((IMObject) context.getContextBean());
                } else {
                    value = context.getValue(getPath());
                }
            }
        }

        return value;
    }

    /**
     * Check whether this node is a boolean type.
     *
     * @return boolean
     */
    @Override
    public boolean isBoolean() {
        Class<?> clazz = getClassType();
        return (Boolean.class == clazz) || (boolean.class == clazz);
    }

    /**
     * Check whether this node is a collection
     *
     * @return boolean
     */
    @Override
    public boolean isCollection() {
        Class<?> clazz = getClassType();
        return ((clazz != null) && (Collection.class.isAssignableFrom(clazz)
                                    || Map.class.isAssignableFrom(clazz)
                                    || PropertyCollection.class.isAssignableFrom(clazz)));
    }

    /**
     * Cast the input value as a collection. If this can't be done then
     * throw and exception
     *
     * @param object the object to cast
     * @return Collection
     * the returned collection object
     * @throws DescriptorException if the cast cannot be made
     */
    public Collection<?> toCollection(Object object) {
        Collection<?> collection = null;

        if (object == null) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.CannotCastToCollection);
        } else if (object instanceof Collection) {
            collection = (Collection<?>) object;
        } else if (object instanceof PropertyCollection) {
            collection = ((PropertyCollection) object).values();
        } else if (object instanceof Map) {
            collection = ((Map<?, ?>) object).values();
        }

        return collection;
    }

    /**
     * Indicates if this node is acomplex node. If the node has an
     * archetypeRange assertion or the node has a cardinality > 1 then the node
     * is deemed to be a complex node
     *
     * @return boolean true if complex
     */
    @Override
    public boolean isComplexNode() {
        return (getMaxCardinality() == UNBOUNDED)
               || (getMaxCardinality() > 1)
               || (containsAssertionType("archetypeRange"));
    }

    /**
     * Check whether this node a date type.
     *
     * @return boolean
     */
    @Override
    public boolean isDate() {
        Class<?> clazz = getClassType();
        return Date.class == clazz;

    }

    /**
     * @return Returns the isDerived.
     */
    @Override
    public boolean isDerived() {
        return isDerived;
    }

    /**
     * @return Returns the isHidden.
     */
    @Override
    public boolean isHidden() {
        return isHidden;
    }

    /**
     * Indicates that this node defines an identifier. When creating the nodes
     * for an Archetype a node should be added for the generic IMObject uid
     * identifier. This will allow the id to be displayed when viewing or
     * editing the Archetyped object even though the Id is not definined as a
     * real node in the Archetype.
     *
     * @return boolean
     */
    public boolean isIdentifier() {
        return getName().equals(IDENTIFIER_NODE_NAME);

    }

    /**
     * Check whether the node maximum length is large. Should be set to true for
     * string nodes where the display length > DEFAULT_DISPLAY_LENGTH.
     * Presentation layer will utilise this to decide whether to display as
     * TextField or TextArea.
     *
     * @return boolean
     */
    public boolean isLarge() {
        return getMaxLength() > DEFAULT_MAX_LENGTH;
    }

    /**
     * Check whether this node is a lookup
     *
     * @return boolean
     */
    @Override
    public boolean isLookup() {
        boolean result = false;
        for (String key : assertionDescriptors.keySet()) {
            if (key.startsWith("lookup")) {
                result = true;
                break;
            }
        }
        return result;
    }

    /**
     * Check whether this ia a money type
     *
     * @return boolean
     */
    @Override
    public boolean isMoney() {
        return getClassType() == Money.class;
    }

    /**
     * Check whether this node is a numeric type.
     *
     * @return boolean
     */
    @Override
    public boolean isNumeric() {
        Class<?> clazz = getClassType();
        return (clazz != null) && ((Number.class.isAssignableFrom(clazz)) || (byte.class == clazz)
                                   || (short.class == clazz) || (int.class == clazz)
                                   || (long.class == clazz) || (float.class == clazz)
                                   || (double.class == clazz));
    }

    /**
     * Check whether this node is an object reference. An object reference is a
     * node that references another oject subclassed from IMObject.
     *
     * @return boolean
     */
    @Override
    public boolean isObjectReference() {
        return IMObjectReference.class.isAssignableFrom(getClassType());
    }

    /**
     * This is a convenience method that checks whether there is a parent child
     * relationship within this node. A parent child relationship only
     * applicable for node descriptors that reference a collection.
     *
     * @return boolean
     */
    @Override
    public boolean isParentChild() {
        return isParentChild && isCollection();
    }

    /**
     * This method indicates that this node descriptor is read-only.
     *
     * @return {@code true} if the node descriptor is read only
     */
    @Override
    public boolean isReadOnly() {
        return isReadOnly;
    }

    /**
     * Check whether this node is mandatory.
     *
     * @return boolean
     */
    @Override
    public boolean isRequired() {
        return getMinCardinality() > 0;
    }

    /**
     * Check whether this node is a string
     *
     * @return boolean
     */
    @Override
    public boolean isString() {
        return String.class == getClassType();
    }

    /**
     * A helper method that checks to see if the specified {@link IMObject}
     * matches the specified filter. This is only  relevant for collection
     * nodes
     *
     * @param imobj the object to check
     * @return boolean
     * true if it matches the filter
     */
    public boolean matchesFilter(IMObject imobj) {
        boolean matches = false;

        if (isCollection()) {
            if (StringUtils.isEmpty(filter)) {
                matches = true;
            } else {
                String shortName = imobj.getArchetype();
                matches = shortName.matches(modFilter);
            }
        }

        return matches;
    }

    /**
     * Delete the specified assertion descriptor
     *
     * @param descriptor the assertion to delete
     */
    @Override
    public void removeAssertionDescriptor(org.openvpms.component.model.archetype.AssertionDescriptor descriptor) {
        assertionDescriptors.remove(descriptor.getName());
    }

    /**
     * Delete the assertion descriptor with the specified type
     *
     * @param type the type name
     */
    @Override
    public void removeAssertionDescriptor(String type) {
        assertionDescriptors.remove(type);
    }

    /**
     * Remove the specified child object from the collection defined by this
     * node descriptor using the nominated {@link IMObject} as the root context.
     * <p>
     * If this node descriptor is not of type collection, or the context object
     * is null it will raise an exception.
     *
     * @param context the root context object
     * @param child   the child element to remove
     * @throws DescriptorException if it fails to complete this request
     */
    public void removeChildFromCollection(IMObject context, Object child) {
        if (context == null) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.FailedToRemoveChildElement,
                    getName());
        }

        if (!isCollection()) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.FailedToRemoveChildElement,
                    getName());
        }

        Object obj = JXPathHelper.newContext(context).getValue(getPath());

        try {
            if (StringUtils.isEmpty(baseName)) {
                // no base name specified look at the type to determine
                // what method to call
                Class<?> clazz = getClassType();
                if (Collection.class.isAssignableFrom(clazz)) {
                    MethodUtils.invokeMethod(obj, "remove", child);
                } else if (Map.class.isAssignableFrom(clazz)) {
                    MethodUtils.invokeMethod(obj, "remove", child);
                } else {
                    throw new DescriptorException(
                            DescriptorException.ErrorCode.FailedToRemoveChildElement,
                            getName());
                }
            } else {
                // if a baseName has been specified then prepend 'add' to the
                // base name and execute the derived method on contxt object
                String methodName = "remove" + StringUtils.capitalize(baseName);

                if (obj instanceof IMObject) {
                    MethodUtils.invokeMethod(obj, methodName, child);
                } else {
                    MethodUtils.invokeMethod(context, methodName, child);
                }

            }
        } catch (Exception exception) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.FailedToRemoveChildElement,
                    exception, getName());
        }
    }

    /**
     * @param baseName The baseName to set.
     */
    @Override
    public void setBaseName(String baseName) {
        this.baseName = baseName;
    }

    /**
     * @param defaultValue The defaultValue to set.
     */
    @Override
    public void setDefaultValue(String defaultValue) {
        this.defaultValue = defaultValue;
    }

    /**
     * @param isDerived The isDerived to set.
     */
    @Override
    public void setDerived(boolean isDerived) {
        this.isDerived = isDerived;
    }

    /**
     * @param derivedValue The derivedValue to set.
     */
    @Override
    public void setDerivedValue(String derivedValue) {
        this.derivedValue = derivedValue;
    }

    /**
     * @param displayName The displayName to set.
     */
    @Override
    public void setDisplayName(String displayName) {
        this.displayName = displayName;
    }

    /**
     * @param filter The filter to set.
     */
    @Override
    public void setFilter(String filter) {
        this.filter = filter;
        if (!StringUtils.isEmpty(filter)) {
            this.modFilter = filter.replace("*", ".*");
        }
    }

    /**
     * @param isHidden The isHidden to set.
     */
    @Override
    public void setHidden(boolean isHidden) {
        this.isHidden = isHidden;
    }

    /**
     * @param index The index to set.
     */
    @Override
    public void setIndex(int index) {
        this.index = index;
    }

    /**
     * @param maxCardinality The maxCardinality to set.
     */
    @Override
    public void setMaxCardinality(int maxCardinality) {
        this.maxCardinality = maxCardinality;
    }

    /**
     * This setter enabled the user to specify an unbounded maximum collection
     * using '*'.
     * <br/>
     * NOTE: this is used by castor serialisation
     *
     * @param maxCardinality The maxCardinality to set.
     */
    public void setMaxCardinalityAsString(String maxCardinality) {
        if (maxCardinality.equals(UNBOUNDED_AS_STRING)) {
            setMaxCardinality(UNBOUNDED);
        } else {
            setMaxCardinality(Integer.parseInt(maxCardinality));
        }
    }

    /**
     * @param maxLength The maxLength to set.
     */
    @Override
    public void setMaxLength(int maxLength) {
        this.maxLength = maxLength;
    }

    /**
     * @param minCardinality The minCardinality to set.
     */
    @Override
    public void setMinCardinality(int minCardinality) {
        this.minCardinality = minCardinality;
    }

    /**
     * @param minLength The minLength to set.
     */
    @Override
    public void setMinLength(int minLength) {
        this.minLength = minLength;
    }

    /**
     * NOTE: this is used by castor serialisation
     *
     * @param nodes The nodeDescriptors to set.
     */
    public void setNodeDescriptorsAsArray(NodeDescriptor[] nodes) {
        this.nodeDescriptors = new LinkedHashMap<>();
        int i = 0;
        for (NodeDescriptor node : nodes) {
            node.setIndex(i++);
            addNodeDescriptor(node);
        }
    }

    /**
     * @param parentChild The parentChild to set.
     */
    @Override
    public void setParentChild(boolean parentChild) {
        this.isParentChild = parentChild;
    }

    /**
     * @param path The path to set.
     */
    @Override
    public void setPath(String path) {
        this.path = path;
    }

    /**
     * Set the value of the readOnly attribute
     *
     * @param value if {@code true} marks the node descriptor read-only
     */
    @Override
    public void setReadOnly(boolean value) {
        this.isReadOnly = value;
    }

    /**
     * @param type The type to set.
     */
    public void setType(String type) {
        if (StringUtils.isEmpty(type)) {
            this.type = null;
        } else {
            if (type.endsWith("[]")) {
                this.isArray = true;
                this.type = type.substring(0, type.indexOf("[]"));
            } else {
                this.type = type;
            }
        }
        synchronized (this) {
            clazz = null;
        }
    }

    /**
     * Set the node value for the specified {@link IMObject}.
     *
     * @param object the object
     * @param value  the value to set
     * @throws DescriptorException if it cannot set the value
     */
    public void setValue(org.openvpms.component.model.object.IMObject object, Object value) {
        if (object == null) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.NullContextForSetValue,
                    getName());
        }

        setValue(object, JXPathHelper.newContext(object), value);
    }

    /**
     * Set the node value for the specified {@link IMObject}.
     *
     * @param object  the object
     * @param context the JXPath context. Must refer to {@code object}
     * @param value   the value to set
     * @throws DescriptorException if it cannot set the value
     */
    public void setValue(org.openvpms.component.model.object.IMObject object, JXPathContext context, Object value) {
        try {
            for (AssertionDescriptor descriptor : getAssertionDescriptorsAsArray()) {
                value = descriptor.set(value, (IMObject) object, this);
            }
            if (isArray) {
                context.setValue(getPath(), value);
            } else {
                context.setValue(getPath(), transform(value));
            }
        } catch (Exception exception) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.FailedToSetValue,
                    exception, getName());
        }
    }

    /**
     * Returns the archetype descriptor that this is a node of.
     *
     * @return the archetype descriptor that this is a node of. May be
     * {@code null}
     */
    @Override
    public ArchetypeDescriptor getArchetypeDescriptor() {
        return archetype;
    }

    /**
     * Returns the parent node descriptor.
     *
     * @return the parent node descriptor or {@code null}, if this node
     * has no parent.
     */
    public NodeDescriptor getParent() {
        return parent;
    }

    /**
     * Sets the archetype descriptor.
     *
     * @param descriptor the archetype descriptor
     */
    public void setArchetypeDescriptor(ArchetypeDescriptor descriptor) {
        archetype = descriptor;
    }

    /**
     * Sets the parent node descriptor.
     *
     * @param parent the parent node descriptor, or {@code null} if this
     *               node has no parent
     */
    public void setParent(NodeDescriptor parent) {
        this.parent = parent;
    }

    /**
     * Filter the children in the list and return only those that comply with
     * the filter term
     *
     * @param children the initial list of children
     * @return List<IMObject>
     * the filtered set
     */
    private List<org.openvpms.component.model.object.IMObject> filterChildren(
            List<org.openvpms.component.model.object.IMObject> children) {
        // if no filter was specified return the complete list
        if (StringUtils.isEmpty(filter)) {
            return children;
        }

        // otherwise filter
        List<org.openvpms.component.model.object.IMObject> filteredSet = new ArrayList<>();
        for (org.openvpms.component.model.object.IMObject obj : children) {
            if (obj.isA(filter)) {
                filteredSet.add(obj);
            }
        }
        return filteredSet;
    }

    /**
     * This is a helper method that will attempt to convert a string to the
     * type specified by this node descriptor. If the node descriptor is of
     * type string then it will simply return the same string otherwise it
     * will search for a constructor of that type that takes a string and
     * return the transformed object.
     *
     * @param value the string value
     * @return Object
     * the transformed object
     */
    private Object transform(Object value) {
        if ((value == null) ||
            (this.isCollection()) ||
            (value.getClass() == getClassType())) {
            return value;
        }

        try {
            return CONVERTER.convert(value, getClassType());
        } catch (JXPathTypeConversionException exception) {
            throw new DescriptorException(
                    DescriptorException.ErrorCode.FailedToCoerceValue,
                    value.getClass().getName(), getClassType().getName());
        }
    }

    /**
     * Helper to return the value of a named property.
     *
     * @param properties the properties
     * @param name       the property name
     * @return the property value. May be {@code null}
     */
    private String getValue(Map<String, NamedProperty> properties, String name) {
        NamedProperty property = properties.get(name);
        return (property != null) ? (String) property.getValue() : null;
    }

    /**
     * This comparator is used to compare the indices of AssertionDescriptors
     */
    private static class AssertionDescriptorIndexComparator
            implements Comparator<org.openvpms.component.model.archetype.AssertionDescriptor> {

        /* (non-Javadoc)
         * @see java.util.Comparator#compare(T, T)
         */
        public int compare(org.openvpms.component.model.archetype.AssertionDescriptor no1,
                           org.openvpms.component.model.archetype.AssertionDescriptor no2) {
            if (no1 == no2) {
                return 0;
            }
            return Integer.compare(no1.getIndex(), no2.getIndex());
        }
    }

}

