/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.component.business.service.archetype.helper;

import org.openvpms.component.business.domain.im.archetype.descriptor.ArchetypeDescriptor;
import org.openvpms.component.business.domain.im.archetype.descriptor.NodeDescriptor;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.component.service.lookup.LookupService;
import org.openvpms.component.system.common.util.PropertyState;

import java.util.List;

import static org.openvpms.component.business.service.archetype.helper.PropertyResolverException.ErrorCode.InvalidNode;
import static org.openvpms.component.business.service.archetype.helper.PropertyResolverException.ErrorCode.InvalidObject;


/**
 * A {@link PropertyResolver} that resolves node values given a root {@link IMObject}.
 *
 * @author Tim Anderson
 */
public class NodeResolver extends BasePropertyResolver {

    /**
     * The root object.
     */
    private final IMObject root;

    /**
     * The archetype descriptor.
     */
    private final ArchetypeDescriptor archetype;

    /**
     * Constructs a {@link NodeResolver}.
     *
     * @param root    the root object
     * @param service the archetype service
     */
    public NodeResolver(IMObject root, ArchetypeService service) {
        this(root, service, null);
    }

    /**
     * Constructs a {@link NodeResolver}.
     *
     * @param root    the root object
     * @param service the archetype service
     * @param lookups the lookup service. May be {@code null}
     */
    public NodeResolver(IMObject root, ArchetypeService service, LookupService lookups) {
        super(service, lookups);
        this.root = root;
        archetype = (ArchetypeDescriptor) service.getArchetypeDescriptor(root.getArchetype());
    }

    /**
     * Returns the archetype of the root object.
     *
     * @return the archetype of the root object
     */
    public ArchetypeDescriptor getArchetype() {
        return archetype;
    }

    /**
     * Resolves the named property.
     *
     * @param name the property name
     * @return the corresponding property
     * @throws PropertyResolverException if the name is invalid
     */
    public Object getObject(String name) {
        return resolve(name).getValue();
    }

    /**
     * Returns all objects matching the named property.
     * <p/>
     * Unlike {@link #getObject(String)}, this method handles collections of arbitrary size.
     *
     * @param name the property name
     * @return the corresponding property values
     * @throws PropertyResolverException if the name is invalid
     */
    @Override
    public List<Object> getObjects(String name) {
        return getObjects(root, name);
    }

    /**
     * Resolves the state corresponding to a property.
     *
     * @param name the property name
     * @return the resolved state
     * @throws PropertyResolverException if the name is invalid
     */
    @Override
    public PropertyState resolve(String name) {
        PropertyState state;
        IMObject object = root;
        ArchetypeDescriptor currentArchetype = archetype;
        int index;
        while ((index = name.indexOf(".")) != -1) {
            String nodeName = name.substring(0, index);
            NodeDescriptor node = getNode(currentArchetype, nodeName);
            if (node == null) {
                throw new PropertyResolverException(InvalidNode, name, currentArchetype);
            }
            Object value = getValue(object, node, true);
            if (value == null) {
                // object missing.
                object = null;
                break;
            } else if (!(value instanceof IMObject)) {
                throw new PropertyResolverException(InvalidObject, name);
            }
            object = (IMObject) value;
            currentArchetype = getArchetype(object);
            name = name.substring(index + 1);
        }
        if (object != null) {
            state = getLeafPropertyState(object, name, currentArchetype);
        } else {
            state = new PropertyState();
        }
        return state;
    }

}
