/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.component.business.service.archetype.assertion;

import org.openvpms.component.model.object.IMObject;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * Base class for {@link IMObject} assertions.
 *
 * @author Tim Anderson
 */
public class AbstractIMObjectAssertions {

    /**
     * Validates that each active object is of a different type.
     *
     * @param entries the objects to validate
     * @return {@code true} if the objects are a unique type, otherwise {@code false}
     */
    protected <T extends IMObject> boolean validateUniqueType(Collection<T> entries) {
        boolean result = true;
        if (entries.size() > 1) {
            Set<String> types = new HashSet<>();
            for (T entry : entries) {
                if (entry.isActive()) {
                    if (!types.add(entry.getArchetype())) {
                        result = false;
                        break;
                    }
                }
            }
        }
        return result;
    }

    /**
     * Helper to verify that an object is a collection of {@link IMObject} relationships.
     *
     * @param object the object to check
     * @return the collection, or {@code null}, if {@code object} is invalid
     */
    @SuppressWarnings("unchecked")
    protected <T extends IMObject> Collection<T> getObjects(Object object, Class<T> type) {
        if (object instanceof Collection) {
            Collection<?> list = (Collection<?>) object;
            for (Object element : list) {
                if (!type.isAssignableFrom(element.getClass())) {
                    return null;
                }
            }
            return (Collection<T>) object;
        }
        return null;
    }
}
