/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.component.business.service.lookup;

import org.openvpms.component.business.dao.im.common.IMObjectDAO;
import org.openvpms.component.business.domain.im.archetype.descriptor.ArchetypeDescriptor;
import org.openvpms.component.business.domain.im.archetype.descriptor.NodeDescriptor;
import org.openvpms.component.business.domain.im.common.IMObject;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.business.service.archetype.helper.TypeHelper;
import org.openvpms.component.business.service.archetype.helper.lookup.LookupAssertion;
import org.openvpms.component.business.service.archetype.helper.lookup.LookupAssertionFactory;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.lookup.LookupRelationship;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.Constraints;
import org.openvpms.component.system.common.query.IMObjectQueryIterator;
import org.openvpms.component.system.common.query.NodeConstraint;
import org.openvpms.component.system.common.query.NodeSortConstraint;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;


/**
 * Abstract implementation of the {@link ILookupService}, that delegates to the {@link IArchetypeService}.
 *
 * @author Tim Anderson
 */
public abstract class AbstractLookupService implements ILookupService {

    /**
     * The archetype service
     */
    private final IArchetypeService service;

    /**
     * The data access object.
     */
    private final IMObjectDAO dao;

    /**
     * The default lookup node.
     */
    private static final String DEFAULT_LOOKUP = "defaultLookup"; // NON-NLS


    /**
     * Constructs an {@link AbstractLookupService}.
     *
     * @param service the archetype service
     * @param dao     the data access object
     */
    protected AbstractLookupService(IArchetypeService service, IMObjectDAO dao) {
        this.service = service;
        this.dao = dao;
    }

    /**
     * Returns the active lookup with the specified lookup archetype short name and code.
     *
     * @param shortName the lookup archetype short name
     * @param code      the lookup code
     * @return the corresponding lookup or {@code null} if none is found
     */
    @Override
    public Lookup getLookup(String shortName, String code) {
        return getLookup(shortName, code, true);
    }

    /**
     * Returns the lookup with the specified lookup archetype short name and code.
     *
     * @param shortName   the lookup archetype short name. May contain wildcards
     * @param code        the lookup code
     * @param activeOnly, if {@code true}, the lookup must be active, otherwise it must be active/inactive
     * @return the corresponding lookup or {@code null} if none is found
     */
    @Override
    public Lookup getLookup(String shortName, String code, boolean activeOnly) {
        return query(shortName, code, activeOnly);
    }

    /**
     * Returns all active lookups with the specified lookup archetype short name.
     *
     * @param shortName the lookup archetype short name
     * @return a collection of lookups with the specified short name
     */
    @Override
    public Collection<Lookup> getLookups(String shortName) {
        return query(shortName);
    }

    /**
     * Returns the default lookup for the specified lookup archetype short name.
     *
     * @param shortName the lookup archetype short name
     * @return the default lookup, or {@code null} if none is found
     */
    @Override
    public Lookup getDefaultLookup(String shortName) {
        ArchetypeQuery query = new ArchetypeQuery(shortName, false, true)
                .add(Constraints.eq(DEFAULT_LOOKUP, true))
                .setMaxResults(1);
        List<IMObject> results = getService().get(query).getResults();

        return (!results.isEmpty()) ? (Lookup) results.get(0) : null;
    }

    /**
     * Returns the lookups that are the source of any lookup relationship where
     * the supplied lookup is the target.
     *
     * @param lookup the target lookup
     * @return a collection of source lookups
     */
    @Override
    public Collection<Lookup> getSourceLookups(Lookup lookup) {
        return getSourceLookups(lookup.getTargetLookupRelationships());
    }

    /**
     * Returns the lookups that are the source of specific lookup relationships
     * where the supplied lookup is the target.
     *
     * @param lookup                the target lookup
     * @param relationshipShortName the relationship short name. May contain
     *                              wildcards
     * @return a collection of source lookups
     */
    @Override
    public Collection<Lookup> getSourceLookups(Lookup lookup, String relationshipShortName) {
        Collection<LookupRelationship> relationships
                = getRelationships(relationshipShortName,
                                   lookup.getTargetLookupRelationships());
        return getSourceLookups(relationships);
    }

    /**
     * Returns the lookups that are the target of any lookup relationship where
     * the supplied lookup is the source.
     *
     * @param lookup the source lookup
     * @return a collection of target lookups
     */
    @Override
    public Collection<Lookup> getTargetLookups(Lookup lookup) {
        return getTargetLookups(lookup.getSourceLookupRelationships());
    }

    /**
     * Returns the lookups that are the target of specific lookup relationships
     * where the supplied lookup is the source.
     *
     * @param lookup                the source lookup
     * @param relationshipShortName the relationship short name. May contain
     *                              wildcards
     * @return a collection of target lookups
     */
    @Override
    public Collection<Lookup> getTargetLookups(Lookup lookup, String relationshipShortName) {
        Collection<LookupRelationship> relationships
                = getRelationships(relationshipShortName,
                                   lookup.getSourceLookupRelationships());
        return getTargetLookups(relationships);
    }

    /**
     * Returns a list of lookups for an archetype's node.
     *
     * @param shortName the archetype short name
     * @param node      the node name
     * @return a list of lookups
     */
    @Override
    public Collection<Lookup> getLookups(String shortName, String node) {
        ArchetypeDescriptor archetype = service.getArchetypeDescriptor(shortName);
        if (archetype == null) {
            throw new IllegalArgumentException("Invalid archetype shortname: " + shortName);
        }
        NodeDescriptor descriptor = archetype.getNodeDescriptor(node);
        if (descriptor == null) {
            throw new IllegalArgumentException("Invalid node name: " + node);
        }
        LookupAssertion assertion = LookupAssertionFactory.create(descriptor, service, this);
        return assertion.getLookups();
    }

    /**
     * Return a list of lookups for a given object and node value.
     * <p>
     * Inactive lookups will be excluded, unless they are explicitly referred to.<br/>
     * This will limit lookups returned if the node refers to the source or target of a lookup relationship.
     *
     * @param object the object
     * @param node   the node name
     * @return a list of lookups
     */
    @Override
    public Collection<Lookup> getLookups(org.openvpms.component.model.object.IMObject object, String node) {
        IMObjectBean bean = service.getBean(object);
        NodeDescriptor descriptor = (NodeDescriptor) bean.getNode(node);
        if (descriptor == null) {
            throw new IllegalArgumentException("Invalid node name: " + node);
        }
        LookupAssertion assertion = LookupAssertionFactory.create(descriptor, service, this);
        return assertion.getLookups(object);
    }

    /**
     * Returns a lookup based on the value of a node. The lookup may be inactive.
     *
     * @param object the object
     * @param node   the node name
     * @return the lookup, or {@code null} if none is found
     */
    @Override
    public Lookup getLookup(org.openvpms.component.model.object.IMObject object, String node) {
        IMObjectBean bean = service.getBean(object);
        return bean.getLookup(node);
    }

    /**
     * Returns a lookup's name based on the value of a node. The lookup may be inactive.
     *
     * @param object the object
     * @param node   the node name
     * @return the lookup's name, or {@code null} if none is found
     */
    @Override
    public String getName(org.openvpms.component.model.object.IMObject object, String node) {
        Lookup lookup = getLookup(object, node);
        return (lookup != null) ? lookup.getName() : null;
    }

    /**
     * Returns the lookup for an archetype's node and the specified code.
     *
     * @param archetype the archetype
     * @param node      the node name
     * @param code      the lookup code
     * @return the lookup or {@code null} if none is found
     */
    @Override
    public Lookup getLookup(String archetype, String node, String code) {
        Collection<Lookup> lookups = getLookups(archetype, node);
        for (Lookup lookup : lookups) {
            if (lookup.getCode().equals(code)) {
                return lookup;
            }
        }
        return null;
    }

    /**
     * Returns the lookup's name for an archetype's node and the specified code.
     *
     * @param archetype the archetype
     * @param node      the node name
     * @param code      the lookup code
     * @return the lookup's name or {@code null} if none is found
     */
    @Override
    public String getName(String archetype, String node, String code) {
        Lookup lookup = getLookup(archetype, node, code);
        return (lookup != null) ? lookup.getName() : null;
    }

    /**
     * Replaces one lookup with another.
     * <p>
     * Each lookup must be of the same archetype.
     *
     * @param source the lookup to replace
     * @param target the lookup to replace {@code source} with
     */
    @Override
    public void replace(Lookup source, Lookup target) {
        if (!source.getArchetype().equals(target.getArchetype())) {
            throw new LookupServiceException(LookupServiceException.ErrorCode.CannotReplaceArchetypeMismatch);
        }
        dao.replace(source, target);
    }

    /**
     * Returns the archetype service.
     *
     * @return the archetype service
     */
    protected IArchetypeService getService() {
        return service;
    }

    /**
     * Returns the DAO.
     *
     * @return the DAO
     */
    protected IMObjectDAO getDAO() {
        return dao;
    }

    /**
     * Returns the lookup with the specified lookup archetype short name and
     * code.
     *
     * @param shortName the lookup archetype short name
     * @param code      the lookup code
     * @return the corresponding lookup or {@code null} if none is found
     */
    protected Lookup query(String shortName, String code, boolean activeOnly) {
        ArchetypeQuery query = new ArchetypeQuery(shortName, false, activeOnly);
        query.add(new NodeConstraint("code", code)); // NON-NLS
        query.setMaxResults(1);
        List<IMObject> results = service.get(query).getResults();
        return (!results.isEmpty()) ? (Lookup) results.get(0) : null;
    }

    /**
     * Returns all active lookups with the specified lookup archetype short name.
     *
     * @param shortName the lookup archetype short name
     * @return a collection of lookups with the specified short name
     */
    protected Collection<Lookup> query(String shortName) {
        ArchetypeQuery query = new ArchetypeQuery(shortName, false, true);
        query.setMaxResults(1000);
        query.add(new NodeSortConstraint("id"));
        Iterator<Lookup> iter = new IMObjectQueryIterator<>(service, query);
        List<Lookup> result = new ArrayList<>();
        while (iter.hasNext()) {
            result.add(iter.next());
        }
        return result;
    }

    /**
     * Retrieves a lookup by reference.
     *
     * @param reference the lookup reference. May be {@code null}
     * @return the corresponding lookup, or {@code null} if none is found. The lookup may be inactive
     */
    protected Lookup getLookup(Reference reference) {
        return (reference != null) ? (Lookup) service.get(reference) : null;
    }

    /**
     * Returns all source lookups of the specified relationships.
     *
     * @param relationships the relationships
     * @return the source lookups
     */
    private Collection<Lookup> getSourceLookups(
            Collection<LookupRelationship> relationships) {
        Collection<Lookup> result;
        if (!relationships.isEmpty()) {
            result = new ArrayList<>();
            for (LookupRelationship relationship : relationships) {
                Lookup source = getLookup(relationship.getSource());
                if (source != null) {
                    result.add(source);
                }
            }
        } else {
            result = Collections.emptyList();
        }
        return result;
    }

    /**
     * Returns all target lookups of the specified relationships.
     *
     * @param relationships the relationships
     * @return the target lookups
     */
    private Collection<Lookup> getTargetLookups(
            Collection<LookupRelationship> relationships) {
        Collection<Lookup> result;
        if (!relationships.isEmpty()) {
            result = new ArrayList<>();
            for (LookupRelationship relationship : relationships) {
                Lookup target = getLookup(relationship.getTarget());
                if (target != null) {
                    result.add(target);
                }
            }
        } else {
            result = Collections.emptyList();
        }
        return result;
    }

    /**
     * Helper to return all relationships matching the specified short name.
     *
     * @param shortName     the relationship short name
     * @param relationships the relatiosnhips to search
     * @return all relationships with the specified short name
     */
    private Collection<LookupRelationship> getRelationships(
            String shortName, Collection<LookupRelationship> relationships) {
        Collection<LookupRelationship> result = null;
        for (LookupRelationship relationship : relationships) {
            if (TypeHelper.isA(relationship, shortName)) {
                if (result == null) {
                    result = new ArrayList<>();
                }
                result.add(relationship);
            }
        }
        if (result == null) {
            result = Collections.emptyList();
        }
        return result;
    }

}
