/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.component.business.service.archetype.helper;

import org.apache.commons.collections.Predicate;
import org.openvpms.component.business.domain.im.common.Entity;
import org.openvpms.component.business.domain.im.common.EntityRelationship;
import org.openvpms.component.business.domain.im.common.IMObject;
import org.openvpms.component.business.domain.im.common.IMObjectReference;
import org.openvpms.component.business.domain.im.common.IMObjectRelationship;
import org.openvpms.component.business.domain.im.common.SequencedPeriodRelationship;
import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.object.Relationship;

import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Set;


/**
 * Helper to access an {@link Entity}'s properties via their names.
 *
 * @author Tim Anderson
 */
public class EntityBean extends IMObjectBean {

    /**
     * Constructs a new {@link EntityBean}.
     *
     * @param entity the entity
     */
    public EntityBean(Entity entity) {
        this(entity, null);
    }

    /**
     * Constructs a new {@link EntityBean}.
     *
     * @param entity  the entity
     * @param service the archetype service
     */
    public EntityBean(Entity entity, IArchetypeService service) {
        super(entity, service);
    }

    /**
     * Returns the underlying entity.
     *
     * @return the underlying entity
     */
    public Entity getEntity() {
        return (Entity) getObject();
    }

    /**
     * Adds an entity relationship to both the source and target entity.
     *
     * @param shortName the relationship short name
     * @param target    the target entity
     * @return the new relationship
     * @throws ArchetypeServiceException for any archetype service error
     */
    public EntityRelationship addRelationship(String shortName, Entity target) {
        Entity entity = getEntity();
        EntityRelationship r = getArchetypeService().create(shortName, EntityRelationship.class);
        r.setSource(entity.getObjectReference());
        r.setTarget(target.getObjectReference());
        entity.addEntityRelationship(r);
        target.addEntityRelationship(r);
        return r;
    }

    /**
     * Adds a new relationship between the current entity (the source), and the supplied target.
     *
     * @param name   the entity relationship node name, used to determine which relationship to create
     * @param target the target entity
     * @return the new relationship
     * @throws ArchetypeServiceException for any archetype service error
     * @throws IMObjectBeanException     if {@code name} is an invalid node, there is no relationship that supports
     *                                   {@code target}, or multiple relationships can support {@code target}
     */
    public EntityRelationship addNodeRelationship(String name, Entity target) {
        return (EntityRelationship) addNodeTarget(name, target);
    }

    /**
     * Adds a new relationship between the current entity (the source), and the supplied target.
     * <p>
     * If the relationship is an {@link EntityRelationship}, it will also be added to the target.
     *
     * @param name   the name
     * @param target the target
     * @return the new relationship
     * @throws ArchetypeServiceException for any archetype service error
     * @throws IMObjectBeanException     if the relationship archetype is not found
     */
    @Override
    public IMObjectRelationship addNodeTarget(String name, IMObject target) {
        IMObjectRelationship result = super.addNodeTarget(name, target);
        if (target instanceof Entity && result instanceof EntityRelationship) {
            ((Entity) target).addEntityRelationship((EntityRelationship) result);
        }
        return result;
    }

    /**
     * Returns the first entity relationship with the specified entity as a target.
     *
     * @param target the target entity
     * @return the first entity relationship with {@code target} as its target or {@code null} if none is found
     */
    public EntityRelationship getRelationship(Entity target) {
        return getRelationship(target.getObjectReference());
    }

    /**
     * Returns the first entity relationship with the specified entity as a
     * target.
     *
     * @param target the target entity
     * @return the first entity relationship with {@code target} as its
     * target or {@code null} if none is found
     */
    public EntityRelationship getRelationship(Reference target) {
        Entity entity = getEntity();
        for (Relationship r : entity.getEntityRelationships()) {
            if (target.equals(r.getTarget())) {
                return (EntityRelationship) r;
            }
        }
        return null;
    }

    /**
     * Returns all active relationships with the specified short name.
     *
     * @param shortName the relationship short name
     * @return all active relationships with the specified short name
     */
    public List<EntityRelationship> getRelationships(String shortName) {
        return getRelationships(shortName, true);
    }

    /**
     * Returns all relationships with the specified short name.
     *
     * @param shortName the relationship short name
     * @param active    determines if the relationships must be active or not
     * @return all relationships with the specified short name
     */
    @SuppressWarnings("unchecked")
    public List<EntityRelationship> getRelationships(String shortName, boolean active) {
        Set<?> relationships = getEntity().getEntityRelationships();
        return select((Set<EntityRelationship>) relationships, getActiveIsA(active, shortName));
    }

    /**
     * Removes an entity relationship.
     *
     * @param relationship the relationship to remove
     */
    public void removeRelationship(EntityRelationship relationship) {
        Entity entity = getEntity();
        entity.removeEntityRelationship(relationship);
    }

    /**
     * Returns the source entity from the first active entity relationship with active source entity, for the specified
     * node.
     *
     * @param node the entity relationship node name
     * @return the source entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeSourceEntity(String node) {
        return (Entity) getNodeSourceObject(node);
    }

    /**
     * Returns the source entity from the first entity relationship for the
     * specified node.
     *
     * @param node   the entity relationship node name
     * @param active determines if the relationship and source entity must be
     *               active
     * @return the source entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeSourceEntity(String node, boolean active) {
        return (Entity) getNodeSourceObject(node, active);
    }

    /**
     * Returns the source entity from the first active entity relationship
     * with active source entity, matching the specified predicate.
     *
     * @param node      the entity relationship node name
     * @param predicate the predicate
     * @return the source entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeSourceEntity(String node, Predicate predicate) {
        return (Entity) getNodeSourceObject(node, predicate);
    }

    /**
     * Returns the source entity from the first entity relationship matching the specified predicate.
     *
     * @param node      the entity relationship node name
     * @param predicate the predicate
     * @param active    determines if the entity must be active or not
     * @return the source entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeSourceEntity(String node, Predicate predicate, boolean active) {
        return (Entity) getNodeSourceObject(node, predicate, active);
    }

    /**
     * Returns the target entity from the first active entity relationship with active target entity, for the specified
     * node.
     *
     * @param node the entity relationship node name
     * @return the target entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeTargetEntity(String node) {
        return (Entity) getNodeTargetObject(node);
    }

    /**
     * Returns the target entity from the first entity relationship for the specified node.
     *
     * @param node   the entity relationship node
     * @param active determines if the relationship and target entity must be active
     * @return the target entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeTargetEntity(String node, boolean active) {
        return (Entity) getNodeTargetObject(node, active);
    }

    /**
     * Returns the target entity from the first active entity relationship with active target entity, for the specified
     * node.
     *
     * @param node      the entity relationship node name
     * @param predicate the predicate
     * @return the target entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeTargetEntity(String node, Predicate predicate) {
        return (Entity) getNodeTargetObject(node, predicate);
    }

    /**
     * Returns the target entity from the first entity relationship for the specified node.
     *
     * @param node      the entity relationship node name
     * @param predicate the predicate
     * @param active    determines if the entity must be active or not
     * @return the target entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeTargetEntity(String node, Predicate predicate, boolean active) {
        return (Entity) getNodeTargetObject(node, predicate, active);
    }

    /**
     * Returns the target entity from the first entity relationship matching the specified short name. The relationship
     * must be active at the specified time, and have an active target entity.
     *
     * @param node the entity relationship node name
     * @param time the time
     * @return the target entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeSourceEntity(String node, Date time) {
        return (Entity) getNodeSourceObject(node, time);
    }

    /**
     * Returns the source entity from the first entity relationship that is active at the specified time, for the
     * specified node.
     *
     * @param node   the entity relationship node
     * @param time   the time
     * @param active determines if the entity must be active
     * @return the source entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeSourceEntity(String node, Date time, boolean active) {
        return (Entity) getNodeSourceObject(node, time, active);
    }

    /**
     * Returns the target entity from the first entity relationship with active target entity that is active at the
     * specified time, for the specified node.
     *
     * @param node the entity relationship node
     * @param time the time
     * @return the target entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeTargetEntity(String node, Date time) {
        return (Entity) getNodeTargetObject(node, time);
    }

    /**
     * Returns the target entity from the first entity relationship that is active at the specified time, for the
     * specified node.
     *
     * @param node   the entity relationship node
     * @param time   the time
     * @param active determines if the entity must be active
     * @return the target entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getNodeTargetEntity(String node, Date time, boolean active) {
        return (Entity) getNodeTargetObject(node, time, active);
    }

    /**
     * Returns the active source entities from each active relationship for the
     * specified node. If a source reference cannot be resolved, it will be
     * ignored.
     *
     * @param node the entity relationship node
     * @return a list of active source entities
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeSourceEntities(String node) {
        return getNodeSourceObjects(node, Entity.class);
    }

    /**
     * Returns the active source entities from each relationship for the
     * specified node that is active at the specified time.
     *
     * @param node the entity relationship node
     * @param time the time
     * @return a list of active source entities
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeSourceEntities(String node, Date time) {
        return getNodeSourceObjects(node, time, Entity.class);
    }

    /**
     * Returns the source entities from each relationship for the specified node
     * that is active at the specified time.
     *
     * @param node   the entity relationship node
     * @param time   the time
     * @param active determines if the entities must be active
     * @return a list of source entities. May contain inactive entities if {@code active} is {@code false}
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeSourceEntities(String node, Date time, boolean active) {
        return getNodeSourceObjects(node, time, active, Entity.class);
    }

    /**
     * Returns the active source entities from each relationship for the
     * specified node that matches the specified predicate.
     *
     * @param node      the entity relationship node
     * @param predicate the predicate
     * @return a list of source entities
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeSourceEntities(String node, Predicate predicate) {
        return getNodeSourceObjects(node, predicate, true, Entity.class);
    }

    /**
     * Returns the source entities from each relationship for the
     * specified node that matches the specified predicate.
     *
     * @param node      the entity relationship node
     * @param predicate the predicate
     * @param active    determines if the entities must be active
     * @return a list of source entities. May contain inactive entities
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeSourceEntities(String node, Predicate predicate, boolean active) {
        return getNodeSourceObjects(node, predicate, active, Entity.class);
    }

    /**
     * Returns the active target entities from each active relationship for the
     * specified node. If a target reference cannot be resolved, it will be
     * ignored.
     *
     * @param node the entity relationship node
     * @return a list of active target entities
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeTargetEntities(String node) {
        return getNodeTargetEntities(node, (Comparator<SequencedPeriodRelationship>) null);
    }

    /**
     * Returns the active target entities from each active relationship for the
     * specified node. If a target reference cannot be resolved, it will be
     * ignored.
     *
     * @param node       the entity relationship node
     * @param comparator if non-null, specifies a comparator to sort relationships
     * @return a list of active target entities
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeTargetEntities(String node, Comparator<SequencedPeriodRelationship> comparator) {
        return getNodeTargetObjects(node, Entity.class, comparator);
    }

    /**
     * Returns the active target entities from each relationship for the
     * specified node that is active at the specified time.
     *
     * @param node the entity relationship node
     * @param time the time
     * @return a list of active target entities
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeTargetEntities(String node, Date time) {
        return getNodeTargetObjects(node, time, Entity.class);
    }

    /**
     * Returns the target entities from each relationship for the specified node
     * that is active at the specified time.
     *
     * @param node   the entity relationship node
     * @param time   the time
     * @param active determines if the entities must be active
     * @return a list of target entities. May contain inactive entities if {@code active} is {@code false}
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeTargetEntities(String node, Date time, boolean active) {
        return getNodeTargetObjects(node, time, active, Entity.class);
    }

    /**
     * Returns the active target entities from each relationship for the
     * specified node that matches the specified predicate.
     *
     * @param node      the entity relationship node
     * @param predicate the predicate
     * @return a list of target entities
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeTargetEntities(String node, Predicate predicate) {
        return getNodeTargetObjects(node, predicate, Entity.class);
    }

    /**
     * Returns the target entities from each relationship for the
     * specified node that matches the specified predicate.
     *
     * @param node      the entity relationship node
     * @param predicate the predicate
     * @param active    determines if the entities must be active
     * @return a list of target entities. May  contain inactive entities
     * @throws ArchetypeServiceException for any archetype service error
     */
    public List<Entity> getNodeTargetEntities(String node, Predicate predicate, boolean active) {
        return getNodeTargetObjects(node, predicate, active, Entity.class);
    }

    /**
     * Returns the source entity references from each active relationship for the specified node.
     *
     * @param node the entity relationship node
     * @return a list of source entity references. May contain references to both active and inactive entities
     */
    public List<IMObjectReference> getNodeSourceEntityRefs(String node) {
        return getNodeSourceObjectRefs(node);
    }

    /**
     * Returns the source entity references from each relationship that is
     * active at the specified time, for the specified node.
     *
     * @param node the entity relationship node
     * @param time the time
     * @return a list of source entity references. May contain references to both active and inactive entities
     */
    public List<IMObjectReference> getNodeSourceEntityRefs(String node, Date time) {
        return getNodeSourceObjectRefs(node, time);
    }

    /**
     * Returns the source entity references from each relationship for the specified node that matches the supplied
     * predicate.
     *
     * @param node      the entity relationship node
     * @param predicate the predicate
     * @return a list of source entity references. May contain references to both active and inactive entities
     */
    public List<IMObjectReference> getNodeSourceEntityRefs(String node, Predicate predicate) {
        return getNodeSourceObjectRefs(node, predicate);
    }

    /**
     * Returns the target entity references from each active relationship for the specified node.
     *
     * @param node the entity relationship node
     * @return a list of target entity references. May contain references to both active and inactive entities
     */
    public List<IMObjectReference> getNodeTargetEntityRefs(String node) {
        return getNodeTargetObjectRefs(node);
    }

    /**
     * Returns the target entity references from each relationship that is active at the specified time, for the
     * specified node.
     *
     * @param node the entity relationship node
     * @param time the time
     * @return a list of target entity references. May contain references to both active and inactive entities
     */
    public List<IMObjectReference> getNodeTargetEntityRefs(String node, Date time) {
        return getNodeTargetObjectRefs(node, time);
    }

    /**
     * Returns the target entity references from each relationship for the specified node that matches the supplied
     * predicate.
     *
     * @param node      the entity relationship node
     * @param predicate the predicate
     * @return a list of target entity references. May contain references to both active and inactive entities
     */
    public List<IMObjectReference> getNodeTargetEntityRefs(String node, Predicate predicate) {
        return getNodeTargetObjectRefs(node, predicate);
    }

    /**
     * Returns all entity relationships for the specified node.
     *
     * @param node the entity relationship node
     * @return a list of relationships
     */
    public List<EntityRelationship> getNodeRelationships(String node) {
        return getValues(node, EntityRelationship.class);
    }

    /**
     * Returns all relationships for the specified node matching the supplied predicate.
     *
     * @param node      the entity relationship node
     * @param predicate the predicate
     * @return a list of relationships matching the predicate
     */
    public List<EntityRelationship> getNodeRelationships(String node, Predicate predicate) {
        return getValues(node, predicate, EntityRelationship.class);
    }

    /**
     * Returns the first relationship for the specified node matching the supplied predicate.
     *
     * @param node      the entity relationship node
     * @param predicate the predicate
     * @return the first relationship matching the predicate
     */
    public EntityRelationship getNodeRelationship(String node, Predicate predicate) {
        return (EntityRelationship) getValue(node, predicate);
    }

    /**
     * Returns the source entity from the first active entity relationship with active source entity, for the specified
     * relationship short name.
     *
     * @param shortName the entity relationship short name
     * @return the source entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getSourceEntity(String shortName) {
        return (Entity) getSourceObject(getEntity().getEntityRelationships(), shortName);
    }

    /**
     * Returns the source entity from the first entity relationship for the specified relationship short name.
     *
     * @param shortName the entity relationship short name
     * @param active    determines if the relationship and entity must be active
     * @return the source entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getSourceEntity(String shortName, boolean active) {
        return (Entity) getSourceObject(getEntity().getEntityRelationships(), shortName, active);
    }

    /**
     * Returns the source entity from the first active entity relationship matching the specified relationship short
     * names and having an active source entity.
     *
     * @param shortNames the entity relationship short names
     * @return the source entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getSourceEntity(String[] shortNames) {
        return (Entity) getSourceObject(getEntity().getEntityRelationships(), shortNames);
    }

    /**
     * Returns the source entity from the first entity relationship matching the specified relationship short names.
     *
     * @param shortNames the entity relationship short names
     * @param active     determines if the relationship and source entity must be active
     * @return the source entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getSourceEntity(String[] shortNames, boolean active) {
        return (Entity) getSourceObject(getEntity().getEntityRelationships(), shortNames, active);
    }

    /**
     * Returns the target entity from the first active entity relationship with active target entity, for the specified
     * relationship short name.
     *
     * @param shortName the entity relationship short names
     * @return the active entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getTargetEntity(String shortName) {
        return (Entity) getTargetObject(getEntity().getEntityRelationships(), shortName);
    }

    /**
     * Returns the target entity from the first entity relationship for the specified relationship short name.
     *
     * @param shortName the entity relationship short names
     * @param active    determines if the relationship and entity must be active
     * @return the entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getTargetEntity(String shortName, boolean active) {
        return (Entity) getTargetObject(getEntity().getEntityRelationships(), shortName, active);
    }

    /**
     * Returns the target entity from the first active entity relationship matching the specified relationship short
     * names and having an active target entity.
     *
     * @param shortNames the entity relationship short names
     * @return the target entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getTargetEntity(String[] shortNames) {
        return (Entity) getTargetObject(getEntity().getEntityRelationships(), shortNames);
    }

    /**
     * Returns the target entity from the first entity relationship matching the specified relationship short names.
     *
     * @param shortNames the entity relationship short names
     * @param active     determines if the relationship and target entity must be active
     * @return the target entity, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getTargetEntity(String[] shortNames, boolean active) {
        return (Entity) getTargetObject(getEntity().getEntityRelationships(), shortNames, active);
    }

    /**
     * Returns the source entity from the first entity relationship matching the specified short name. The relationship
     * must be active at the specified time, and have an active source entity.
     *
     * @param shortName the entity relationship short name
     * @param time      the time
     * @return the source entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getSourceEntity(String shortName, Date time) {
        return (Entity) getSourceObject(getEntity().getEntityRelationships(), shortName, time);
    }

    /**
     * Returns the source entity from the first entity relationship matching the specified short name. The relationship
     * must be active at the specified time.
     *
     * @param shortName the entity relationship short name
     * @param time      the time
     * @param active    determines if the entity must be active
     * @return the source entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getSourceEntity(String shortName, Date time, boolean active) {
        return (Entity) getSourceObject(getEntity().getEntityRelationships(), shortName, time, active);
    }

    /**
     * Returns the source entity from the first entity relationship matching the specified short names. The relationship
     * must be active at the specified time, and have an active source entity.
     *
     * @param shortNames the entity relationship short names
     * @param time       the time
     * @return the source entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getSourceEntity(String[] shortNames, Date time) {
        return (Entity) getSourceObject(getEntity().getEntityRelationships(), shortNames, time);
    }

    /**
     * Returns the source entity from the first entity relationship matching the specified short names. The
     * relationship must be active at the specified time.
     *
     * @param shortNames the entity relationship short names
     * @param time       the time
     * @param active     determines if the entity must be active
     * @return the source entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getSourceEntity(String[] shortNames, Date time, boolean active) {
        return (Entity) getSourceObject(getEntity().getEntityRelationships(), shortNames, time, active);
    }

    /**
     * Returns the source entity from the first entity relationship matching the specified short name. The relationship
     * must be active at the specified time, and have an active target entity.
     *
     * @param shortName the entity relationship short name
     * @param time      the time
     * @return the target entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getTargetEntity(String shortName, Date time) {
        return (Entity) getTargetObject(getEntity().getEntityRelationships(), shortName, time);
    }

    /**
     * Returns the target entity from the first entity relationship matching the specified short name. The relationship
     * must be active at the specified time.
     *
     * @param shortName the entity relationship short name
     * @param time      the time
     * @param active    determines if the entity must be active
     * @return the target entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getTargetEntity(String shortName, Date time, boolean active) {
        return (Entity) getTargetObject(getEntity().getEntityRelationships(), shortName, time, active);
    }

    /**
     * Returns the target entity from the first entity relationship matching the specified short names. The relationship
     * must be active at the specified time, and have an active target entity.
     *
     * @param shortNames the entity relationship short names
     * @param time       the time
     * @return the target entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getTargetEntity(String[] shortNames, Date time) {
        return (Entity) getTargetObject(getEntity().getEntityRelationships(), shortNames, time);
    }

    /**
     * Returns the target entity from the first entity relationship matching the specified short names. The relationship
     * must be active at the specified time.
     *
     * @param shortNames the entity relationship short names
     * @param time       the time
     * @param active     determines if the relationship must be active
     * @return the target entity, or {@code null} if none is found
     * @throws IMObjectBeanException     if the node is invalid
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Entity getTargetEntity(String[] shortNames, Date time, boolean active) {
        return (Entity) getTargetObject(getEntity().getEntityRelationships(), shortNames, time, active);
    }

    /**
     * Returns the source entity reference from the first active entity relationship matching the specified short name.
     *
     * @param shortName the entity relationship short name
     * @return the source reference, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Reference getSourceEntityRef(String shortName) {
        return getSourceObjectRef(getEntity().getEntityRelationships(), shortName);
    }

    /**
     * Returns the source entity reference from the first entity relationship matching the specified short name.
     *
     * @param shortName the entity relationship short name
     * @param active    determines if the relationship must be active
     * @return the source reference, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Reference getSourceEntityRef(String shortName, boolean active) {
        return getSourceObjectRef(getEntity().getEntityRelationships(), shortName, active);
    }

    /**
     * Returns the source entity reference from the first entity relationship matching the specified short names.
     *
     * @param shortNames the entity relationship short names
     * @param active     determines if the relationship must be active
     * @return the source reference, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Reference getSourceEntityRef(String[] shortNames, boolean active) {
        return getSourceObjectRef(getEntity().getEntityRelationships(), shortNames, active);
    }

    /**
     * Returns the target entity reference from the first active entity relationship matching the specified short name.
     *
     * @param shortName the entity relationship short name
     * @return the target reference, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Reference getTargetEntityRef(String shortName) {
        return getTargetObjectRef(getEntity().getEntityRelationships(), shortName);
    }

    /**
     * Returns the target entity reference from the first entity relationship matching the specified short name.
     *
     * @param shortName the entity relationship short name
     * @param active    determines if the relationship must be active
     * @return the target reference, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Reference getTargetEntityRef(String shortName, boolean active) {
        return getTargetObjectRef(getEntity().getEntityRelationships(), shortName, active);
    }

    /**
     * Returns the target entity reference from the first entity relationship matching the specified short name.
     *
     * @param shortNames the entity relationship short names
     * @param active     determines if the relationship must be active
     * @return the target reference, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Reference getTargetEntityRef(String[] shortNames, boolean active) {
        return getTargetObjectRef(getEntity().getEntityRelationships(), shortNames, active);
    }

}
