/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.component.i18n;


import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * Factory for {@link Message}s.
 * <p>
 * Messages are obtained from a resource bundle, with each message having a numeric code.
 *
 * @author Tim Anderson
 */
public class Messages {

    /**
     * The resource bundle path.
     */
    private final String bundlePath;

    /**
     * The group that the messages belong to.
     */
    private final String groupId;

    /**
     * The class loader to use. If not specified, resource bundles will be resolved using the context class loader.
     */
    private final ClassLoader classLoader;

    /**
     * The logger.
     */
    private static final Logger log = LoggerFactory.getLogger(Messages.class);


    /**
     * Constructs a {@link Messages}.
     *
     * @param groupId    the group that the messages belong to
     * @param bundlePath the resource bundle path
     */
    public Messages(String groupId, String bundlePath) {
        this(groupId, bundlePath, null);
    }

    /**
     * Constructs a {@link Messages}.
     *
     * @param groupId the group that the messages belong to
     * @param type    the class used to determine the resource bundle name, and the class loader used to load the bundle
     */
    public Messages(String groupId, Class<?> type) {
        this(groupId, type.getName(), type.getClassLoader());
    }

    /**
     * Constructs a {@link Messages}.
     *
     * @param groupId     the group that the messages belong to
     * @param bundlePath  the resource bundle path
     * @param classLoader the class loader used to load the resource bundle. May be {@code null}
     */
    private Messages(String groupId, String bundlePath, ClassLoader classLoader) {
        this.groupId = groupId;
        this.bundlePath = bundlePath;
        this.classLoader = classLoader;
    }

    /**
     * Creates a new {@link Message} instance for the default locale, containing the formatted message with id
     * {@code code}, from the resource bundle.
     *
     * @param code the message code, corresponding to an entry in the resource bundle
     * @param args the format arguments
     * @return a new message
     */
    public Message create(int code, Object... args) {
        return create(code, Locale.getDefault(), args);
    }

    /**
     * Creates a new {@link Message} instance for the specified locale, containing the formatted message with id
     * {@code code}, from the resource bundle.
     *
     * @param code   the message code, corresponding to an entry in the resource bundle
     * @param locale the locale
     * @param args   the format arguments
     * @return a new message
     */
    public Message create(int code, Locale locale, Object... args) {
        String message = getString(Integer.toString(code), locale, args);
        return new Message(groupId, code, message);
    }

    /**
     * Creates a message that is not read from a resource bundle.
     *
     * @param message the message
     * @return a new message
     */
    public static Message create(String message) {
        return new Message(null, -1, message);
    }

    /**
     * Formats a string using the string with id {@code key}, for the specified locale, from the resource bundle.
     *
     * @param key    the message key, corresponding to an entry in the resource bundle
     * @param locale the locale
     * @param args   the format arguments
     * @return a new message
     */
    protected String getString(String key, Locale locale, Object... args) {
        String result = getValue(key, locale);
        if (result == null) {
            result = formatMissingKey(key, args);
        } else if (args.length != 0) {
            MessageFormat format = new MessageFormat(result, locale);
            try {
                result = format.format(args);
            } catch (Exception exception) {
                result = formatFailed(key, result, args, exception);
            }
        }
        return result;
    }

    /**
     * Returns a value for the specified key and locale.
     *
     * @param key    the key for the desired string
     * @param locale the locale for which a string is desired
     * @return the corresponding value, or {@code null} if its not found
     */
    protected String getValue(String key, Locale locale) {
        ResourceBundle bundle = getResourceBundle(locale);
        String result = null;
        try {
            result = StringUtils.trimToNull(bundle.getString(key));
        } catch (MissingResourceException exception) {
            // ignore
        }
        return result;
    }

    /**
     * Returns the resource bundle for the specified locale.
     *
     * @param locale the locale
     * @return the corresponding resource bundle
     */
    protected ResourceBundle getResourceBundle(Locale locale) {
        return ResourceBundle.getBundle(bundlePath, locale, getClassLoader());
    }

    /**
     * Retrns the class loader to load the resource bundle.
     *
     * @return the class loader
     */
    protected ClassLoader getClassLoader() {
        ClassLoader result = (classLoader != null) ? classLoader : Thread.currentThread().getContextClassLoader();
        return (result != null) ? result : getClass().getClassLoader();
    }

    /**
     * Formats a message when the corresponding resource bundle key doesn't exist.
     * <p>
     * This is to help return some message to the user, in the case where the resource bundle is not in sync with
     * the code that generates the message.
     *
     * @param key  the resource bundle key
     * @param args the arguments to format
     * @return a message
     */
    protected String formatMissingKey(String key, Object... args) {
        log.error("ResourceBundle={} missing key={}", bundlePath, key);
        StringBuilder result = new StringBuilder("?" + key + "?");
        if (args.length != 0) {
            result.append('[');
            result.append(StringUtils.join(args, ','));
            result.append(']');
        }
        return result.toString();
    }

    /**
     * Formats a message when it cannot be formattted by {@code MessageFormat} due to an exception.
     * <p>
     * This is to help return some message to the user, in the case where the resource bundle is not in sync with
     * the code that generates the message.
     *
     * @param key       the resource bundle key
     * @param message   the message
     * @param args      the arguments to format
     * @param exception the cause of the failure
     * @return a message corresponding to the arguments
     */
    protected String formatFailed(String key, String message, Object[] args, Throwable exception) {
        String argText = StringUtils.join(args, ',');
        log.error("Failed to format message, bundle={}, key={}, text='{}', arguments=[{}]", bundlePath, key, message,
                  argText, exception);
        StringBuilder result = new StringBuilder("Message='");
        result.append(message);
        result.append("'");
        if (args.length != 0) {
            result.append(". Arguments=[");
            result.append(argText);
            result.append(']');
        }
        return result.toString();
    }

}
