/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.etl.load;

import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.lookup.LookupRelationship;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.service.lookup.LookupService;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.IArchetypeQuery;
import org.openvpms.component.system.common.query.IMObjectQueryIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;


/**
 * Lookup cache.
 *
 * @author Tim Anderson
 */
class LookupCache {

    /**
     * The archetype service.
     */
    private final IArchetypeService service;

    /**
     * The lookup service.
     */
    private final LookupService lookupService;

    /**
     * A map of lookup relationships short names to the corresponding
     * relationship instances.
     */
    private final Map<String, List<LookupRelationship>> relationshipsByArchetype = new HashMap<>();

    /**
     * The logger.
     */
    private static final Logger log = LoggerFactory.getLogger(LookupCache.class);


    /**
     * Constructs a {@link LookupCache}.
     *
     * @param service the archetype service
     */
    @SuppressWarnings("HardCodedStringLiteral")
    public LookupCache(IArchetypeService service, LookupService lookupService) {
        this.service = service;
        this.lookupService = lookupService;
        if (!(lookupService instanceof CachingLookupService)) {
            log.warn("{} is not configured. The lookup cache may not perform optimally",
                     CachingLookupService.class.getName());
        }
    }

    /**
     * Returns the lookup for a specified lookup archetype short name and code.
     *
     * @param archetype the lookup archetype short name
     * @param code      the lookup code
     * @return the corresponding lookup, or {@code null} if none exists
     * @throws ArchetypeServiceException for any error
     */
    public Lookup get(String archetype, String code) {
        return lookupService.getLookup(archetype, code);
    }

    /**
     * Adds a lookup to the cache.
     *
     * @param lookup the lookup to add
     * @throws ArchetypeServiceException for any error
     */
    public void add(Lookup lookup) {
        if (lookupService instanceof CachingLookupService) {
            ((CachingLookupService) lookupService).add(lookup);
        }
    }

    /**
     * Determines if a lookup exists.
     *
     * @param archetype the lookup archetype short name
     * @param code      the lookup code
     * @return {@code true} if the lookup exists
     * @throws ArchetypeServiceException for any error
     */
    public boolean exists(String archetype, String code) {
        return get(archetype, code) != null;
    }

    /**
     * Adds a lookup relationship to the cache.
     *
     * @param relationship the relationship to add
     * @throws ArchetypeServiceException for any error
     */
    public void add(LookupRelationship relationship) {
        String archetype = relationship.getArchetype();
        List<LookupRelationship> relationships = getRelationships(archetype);
        relationships.add(relationship);
        if (lookupService instanceof CachingLookupService) {
            // need to update the cached lookups with the relationship.
            CachingLookupService cache = (CachingLookupService) lookupService;
            cache.add(relationship);
        }
    }

    /**
     * Determines if a lookup relationship exists.
     *
     * @param archetype the lookup relationship archetype short name
     * @param source    the source lookup reference
     * @param target    the target lookup reference
     * @return {@code true} if the relationship exists
     * @throws ArchetypeServiceException for any error
     */
    public boolean exists(String archetype, Reference source, Reference target) {
        for (LookupRelationship relationship : getRelationships(archetype)) {
            if (Objects.equals(relationship.getSource(), source)
                && Objects.equals(relationship.getTarget(), target)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns all instances of the given lookup relationship archetype.
     *
     * @param archetype the lookup relationship archetype short name
     * @return a list of relationships
     * @throws ArchetypeServiceException for any error
     */
    private List<LookupRelationship> getRelationships(String archetype) {
        List<LookupRelationship> relationships = relationshipsByArchetype.get(archetype);
        if (relationships == null) {
            relationships = new ArrayList<>();
            ArchetypeQuery query = new ArchetypeQuery(archetype, false, true);
            query.setMaxResults(IArchetypeQuery.ALL_RESULTS);
            IMObjectQueryIterator<LookupRelationship> iterator = new IMObjectQueryIterator<>(service, query);
            while (iterator.hasNext()) {
                relationships.add(iterator.next());
            }
            relationshipsByArchetype.put(archetype, relationships);
        }
        return relationships;
    }
}
